/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef IOCONTROLLER_H_
#define IOCONTROLLER_H_

#include "../defines.h"
#include "../data/GraphTypes.h"
#include "Configuration.h"
#include "./pugixml/pugixml.hpp"
#include "../mc/ModelChecker.h"
#include "../mc/SCC_MC.h"
#include "../mc/State_Elim.h"
#include <vector>
#include <map>
#include <string>

namespace scc_cex {

class IOWrapper {
public:
	IOWrapper(std::string str);
	IOWrapper(std::string str, unsigned int input_flags);
	IOWrapper(std::string str, unsigned int input_flags, unsigned int heuristic_flags);
	virtual ~IOWrapper();

	void run();

	bool readFiles();
	Graph<PTYPE>* runModelChecking();
	std::string mcResultToString(Graph<PTYPE>* g) const;
	PTYPE getResultProbability() const;
	
	void setProbModelCheck(PTYPE probability) {
		mProbModelCheck = probability;
	}
		
	/**
	 * Setter
	 * @param id id of target node
	 */
	void setTarget(const unsigned int id) {
		for (unsigned int i = mTargetNodes.size(); i <= id; i++) {
			mTargetNodes.push_back(false);
		}
		mTargetNodes.at(id) = true;
		mTargetIndex = id;
	}

	/**
	 * Getter
	 * @param id id of node
	 * @return true, if target
	 */
	bool isTarget(const unsigned int id) {
		if (id < mTargetNodes.size()) {
			return mTargetNodes.at(id);
		} else {
			return false;
		}
	}

	std::string mResultFileName;
	std::string logFile;


	//former private
	int mInitialNodeID;
	unsigned int mEdgeCount;
	Vertex<PTYPE> *mInitialNode;
	std::vector<bool> mTargetNodes;
	std::vector<Vertex<PTYPE> *> mVertexList;
	std::map<int, AtPropType> mVtxToPropsMap;
	PTYPE mProbModelCheck;
	
private:
	std::istream* mStrm;
	std::string mResultFile;
	std::string inputFile; //stores the path given to the application by the first command line argument

	/** Input flags */
	bool mAtPropsGiven;
	bool mWaitForExternalCalls;
	Graph<PTYPE>* xmlGraph;

	unsigned int mTargetIndex;

	ModelChecker* mModelChecker;

	void init(std::istream *strm);
	void initLogging();

	bool readFiles(std::istream *strm);

	std::string writeMCResult(Graph<PTYPE>* g) const;

	void transformOneInput();
	void transformOneOutput();
	void clearSelfLoops();
};

}

#endif /* IOCONTROLLER_H_ */
