/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "../defines.h"
#include "IOWrapper.h"
#include "input/ProbReader.h"
#include "Configuration.h"

using namespace std;
using namespace param_sccmc_io;

void printHelp(string name) {
	cout << "Parametric SCC based Model Checking" << endl;
	cout << "Copyright (c) RWTH Aachen University 2013" << endl;
	cout << "Authors: Florian Corzilius, Nils Jansen, Matthias Volk" << endl;
	cout << endl;
	cout << "This is Version 1.0. Possible ways to call are: " << endl;
	cout << "./param_sccmc filename.pm [options]" << endl;
	cout << "NOTICE: filepaths have to be relative to the executable!" << endl;
	cout << "  with options:" << endl;
	cout << "--state_elim               (Use state elimination approach)" << endl;
	cout << "--elim_order <index, random, distance, distance_initial, connectivity>" << endl;
	cout << "                           (Elimination order for state elimination, default is distance_initial)" << endl;
	cout << "--elim_decrease            (Elimination order is decreasing, default is decreasing)" << endl;
	cout << "--log <filename>           (Log result as <filename.log>)" << endl;
	cout << "--outputdtmc <filename>    (Save result as <filename.dtmc>)" << endl;
	cout << "--outputxml <filename>     (Save result as <filename.xml>)" << endl;
	cout << endl;
	cout << "This product is distributed under the GPL conditions" << endl;
	cout << "and comes with ABSOLUTELY NO WARRANTY." << endl;
	cout << "This is a free software, and you are welcome to redistribute it." << endl;
	cout << endl;
}

int main(int argc, char* argv[]) {
	if (argc == 1) {
		printHelp(argv[0]);
		return 0;
	}

	scc_cex::ConfigurationSingleton* conf = scc_cex::ConfigurationSingleton::getInstance();
	conf->setIoflIncreasingElimOrder(true);
	conf->setIoflElimOrder(scc_cex::EliminationOrder::INDEX);

	string firstElement = argv[1];

	if (firstElement == "--help") {
		printHelp(argv[0]);
		return 0;
	}

	string lastOption = argv[1];

	string fileName = "";
	
	string targetFile = DEFAULT_RESULT_FILE;
	string logFile = "";
	
	conf->setIoflInputIsFile(true);

	for (int i = 1; i < argc; i++) {
		if (string(argv[i]) == "-o") {
			if (i + 1 < argc)
				targetFile = argv[i + 1];
			else
				targetFile = "NoneSet";
		} else if (string(argv[i]) == "--no_config") {
			conf->setIoflNoConfig(true);
			conf->setIoflOnlyModelChecking(true);
		} else if (string(argv[i]) == "--only_model_checking") {
			conf->setIoflOnlyModelChecking(true);
		} else if (string(argv[i]) == "--xml") {
			conf->setIoflReadFromXml(true);
		} else if (string(argv[i]) == "--state_elim") {
			conf->setIoflStateElim(true);
			//Set default values
			conf->setIoflElimOrder(scc_cex::EliminationOrder::DISTANCE_INITIAL);
			conf->setIoflIncreasingElimOrder(false);
		} else if (string(argv[i]) == "--elim_order") {
			if (i + 1 < argc) {
				string s = argv[i + 1];
				if (s.compare("index") == 0) {
					conf->setIoflElimOrder(scc_cex::EliminationOrder::INDEX);
				} else if (s.compare("random") == 0) {
					conf->setIoflElimOrder(scc_cex::EliminationOrder::RANDOM);
				} else if (s.compare("distance") == 0) {
					conf->setIoflElimOrder(scc_cex::EliminationOrder::DISTANCE);
				} else if (s.compare("distance_initial") == 0) {
					conf->setIoflElimOrder(scc_cex::EliminationOrder::DISTANCE_INITIAL);
				} else if (s.compare("connectivity") == 0) {
					conf->setIoflElimOrder(scc_cex::EliminationOrder::CONNECTIVITY);
				} else {
					//Ignore
					cout << "Elimination order is not valid. Default is index." << endl;
				}
			} else {
				//Ignore
				cout << "No elimination order is given. Default is index." << endl;
			}
			i++;
		} else if (string(argv[i]) == "--elim_decrease") {
			conf->setIoflIncreasingElimOrder(false);
		} else if (string(argv[i]) == "--outputxml") {
			conf->setIoflOutputAsXml(true);
			if (fileName != "") {
				if (i + 1 < argc)
					targetFile = argv[i + 1];
				else
					targetFile = "NoneSet";
			} else {
				string ioInput = "";
				while (ioInput != "o" && ioInput != "i") {
					std::cout << "Is " << argv[i + 1] << " your input or output file? Type 'o' for output and 'i' for input: ";
					getline(cin, ioInput);
				}
				if (ioInput == "o") {
					if (i + 1 < argc)
						targetFile = argv[i + 1];
				} else {
					fileName = argv[i + 1];
				}
			}
			i++;
		} else if (string(argv[i]) == "--outputdtmc") {
			conf->setIoflOutputAsDtmc(true);
			if (fileName != "") {
				if (i + 1 < argc)
					targetFile = argv[i + 1];
				else
					targetFile = "NoneSet";
			} else {
				string ioInput = "";
				while (ioInput != "o" && ioInput != "i") {
					std::cout << "Is " << argv[i + 1] << " your input or output file? Type 'o' for output and 'i' for input: ";
					getline(cin, ioInput);
				}
				if (ioInput == "o") {
					if (i + 1 < argc)
						targetFile = argv[i + 1];
				} else {
					fileName = argv[i + 1];
				}
			}
		} else if (string(argv[i]) == "--log") {
			conf->setIoflLog(true);
			if (i + 1 < argc) {
				logFile = argv[i + 1];
			} else {
				conf->setIoflLog(false);
				std::cout << "No logfile specified. Logging disabled." << endl;
			}
			i++;
		} else if ((string(argv[i]).substr(0, 2) != "--") && (lastOption != "--outputdtmc")
				&& (lastOption != "--outputxml" && (lastOption != "--log")) && (lastOption != "--probBound")) { //|| (string(argv[i]).substr(0,3) == "../") || (string(argv[i]).find_first_of("/") != string::npos)) {
			//primitive filedetector
			fileName = argv[i];
			string fileEnding = fileName.substr(fileName.find_last_of("."), fileName.size());
			if (fileName.find_last_of(".") == (fileName.find_last_of("/") - 1))
				fileEnding = "";
			if (fileEnding != "") {
				if ((fileEnding == ".dtmc") || (fileEnding == ".tra") || (fileEnding == ".pm")) {
					conf->setIoflNoConfig(true);
				}
			}
			if (!scc_cex::doesFileExist(fileName)) {
				cout << "The given file (" << fileName << ") doesn't exist - terminating" << endl;
				return 0;
			}
		} else if (string(argv[i]) == "--bunny") {
			cout << "\\__/)" << endl;
			cout << "(O.o)" << endl;
			cout << "( }{ )" << endl;
			cout << "Das ist Hase. Kopiere Hase und hilf ihm, die Weltherrschaft zu uebernehmen." << endl;
		} else if (string(argv[i]) == "--help") {
			printHelp(argv[0]);
			return 0;
		}

		else if (string(argv[i]) == "--nils") {
			cout << "Nils Jansen is king.";
			return 0;
		} else {
			cout << "There is no argument '" << argv[i] << "'!" << endl;
		}
		lastOption = argv[i];
	}
	
	if (!scc_cex::doesFileExist(fileName)) {
		cout << "The given file (" << fileName << ") doesn't exist - terminating" << endl;
		return 0;
	}

	scc_cex::IOWrapper wrapper(fileName, 0, 0);

	if (targetFile != "") {
		wrapper.mResultFileName = targetFile;
	}

	if (logFile != "") {
		wrapper.logFile = logFile;
	}

	wrapper.run();

	return 0;

}
