/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef DTMCREADER_H_
#define DTMCREADER_H_

#include "InputReader.h"
#include "../../data/GraphTypes.h"
#include "../IOWrapper.h"

using namespace scc_cex;
using namespace std;

namespace param_sccmc_io {

class DTMCReader: public InputReader {
public:
	DTMCReader(IOWrapper *wrapper) {
		this->wrapper = wrapper;
	}
	bool parse(std::istream *strm) const {
		Logger logger = Logger::getInstance("DTMCReader");
		logger.setLogLevel(DEFAULT_LOG_LEVEL);

		if (strm->fail())
			return false;

		int source = -1;
		int target = -1;
		string read_prob = "-1.0";
		PTYPE prob = -1.0;

		string str;
		wrapper->mInitialNodeID = -1;
		int numStates = -1;
		int numTransitions = -1;

		unsigned int numTargets = 0;

		DefaultProbReader<PTYPE> probParser;

		//While loop for extracting header info (number of states etc.)
		//We assume that each line has correct structure,
		//so this loop is not exactly robust!
		while (strm->good()) {

			(*strm) >> str;

			if (str.compare("STATES") == 0) {

				if (numStates != -1) {
					LOG4CPLUS_ERROR(logger, "Multiply defined number of states");
					goto Error;
				}
				(*strm) >> numStates;
				LOG4CPLUS_TRACE(logger, str << " " << numStates);
				if (numStates <= 0) {
					LOG4CPLUS_ERROR(logger, "Number of states must be larger than 0");
					goto Error;
				}
				for (int i = wrapper->mVertexList.size(); i <= numStates; i++) {
					wrapper->mVertexList.push_back(NULL);
				}

			} else if (str.compare("INITIAL") == 0) {

				if (wrapper->mInitialNodeID != -1) {
					LOG4CPLUS_TRACE(logger, "Multiply defined initial node");
					goto Error;
				}
				(*strm) >> wrapper->mInitialNodeID;
				LOG4CPLUS_TRACE(logger, str << " " << wrapper->mInitialNodeID);
				if (wrapper->mInitialNodeID < 0) {
					LOG4CPLUS_ERROR(logger, "Node ids must be non-negative");
					goto Error;
				}
				for (unsigned int i = wrapper->mVertexList.size(); i <= (unsigned int) wrapper->mInitialNodeID; i++) {
					wrapper->mVertexList.push_back(NULL);
				}
				if (wrapper->mVertexList[wrapper->mInitialNodeID] == NULL) {
					wrapper->mVertexList[wrapper->mInitialNodeID] = new Vertex<PTYPE>();
					wrapper->mVertexList[wrapper->mInitialNodeID]->oindex = wrapper->mInitialNodeID;

					map<int, AtPropType>::iterator it = wrapper->mVtxToPropsMap.find(wrapper->mInitialNodeID);
					if (it != wrapper->mVtxToPropsMap.end())
						wrapper->mVertexList[wrapper->mInitialNodeID]->ap = it->second;
				} else {
					assert(wrapper->mVertexList[wrapper->mInitialNodeID]->oindex == (unsigned int ) wrapper->mInitialNodeID);
					LOG4CPLUS_ERROR(logger, "Probability : 1");
					exit(0);
				}
				wrapper->mInitialNode = wrapper->mVertexList[wrapper->mInitialNodeID];

			} else if (str.compare("TARGET") == 0) {
				numTargets++;

				unsigned int id;
				(*strm) >> id;

				LOG4CPLUS_TRACE(logger, str << " " << id);
				if (id < 0) {
					LOG4CPLUS_ERROR(logger, "Node ids must be non-negative");
					goto Error;
				}
				for (unsigned int i = wrapper->mVertexList.size(); i <= id; i++) {
					wrapper->mVertexList.push_back(NULL);
				}
				if (wrapper->mVertexList[id] == NULL) {
					wrapper->mVertexList[id] = new Vertex<PTYPE>();
					wrapper->mVertexList[id]->oindex = id;
					map<int, AtPropType>::iterator it = wrapper->mVtxToPropsMap.find(id);
					if (it != wrapper->mVtxToPropsMap.end())
						wrapper->mVertexList[id]->ap = it->second;
				}
				wrapper->setTarget(id);

			} else if (str.compare("TRANSITIONS") == 0) {

				if (numTransitions != -1) {
					LOG4CPLUS_ERROR(logger, "Multiply defined number of transitions");
					goto Error;
				}
				(*strm) >> numTransitions;
				LOG4CPLUS_TRACE(logger, str << " " << numTransitions);
				wrapper->mEdgeCount = numTransitions;

				if (numTransitions < 0) {
					LOG4CPLUS_ERROR(logger, "Number of transitions must be non-negative");
					goto Error;
				}

			}
			//No more Header info --> next comes a transition
			else {

				source = atoi(str.c_str());
				(*strm) >> target >> read_prob;
				prob = probParser.readProb(read_prob);
				LOG4CPLUS_TRACE(logger, source << " " << target << " " << prob);

				if (!wrapper->isTarget(source)) {
					if (wrapper->mVertexList[source] == NULL) {
						wrapper->mVertexList[source] = new Vertex<PTYPE>();
						wrapper->mVertexList[source]->oindex = source;
						map<int, AtPropType>::iterator it = wrapper->mVtxToPropsMap.find(source);
						if (it != wrapper->mVtxToPropsMap.end())
							wrapper->mVertexList[source]->ap = it->second;
					}
					if (wrapper->mVertexList[target] == NULL) {
						wrapper->mVertexList[target] = new Vertex<PTYPE>();
						wrapper->mVertexList[target]->oindex = target;
						map<int, AtPropType>::iterator it = wrapper->mVtxToPropsMap.find(target);
						if (it != wrapper->mVtxToPropsMap.end())
							wrapper->mVertexList[target]->ap = it->second;
					}
					new Edge<PTYPE>(wrapper->mVertexList[source], wrapper->mVertexList[target], prob);
				}

				//We break out of the loop here, i.e. only the header info and the first
				//transition is read in this while loop
				break;
			}
		}

		if (!strm->good())
			return false;

		//Read all (remaining) transitions
		//We assume that each line has correct structure,
		//so this loop is not exactly robust!
		while (strm->good()) {

			source = -1;
			target = -1;
			read_prob = "-1.0";
			//prob = -1.0;

			(*strm) >> source;
			if (!strm->good())
				break;
			(*strm) >> target;
			if (!strm->good())
				break;
			(*strm) >> read_prob;
			prob = probParser.readProb(read_prob);

			LOG4CPLUS_TRACE(logger, source << " " << target << " " << prob);

			if (!wrapper->isTarget(source)) {
				if (wrapper->mVertexList[source] == NULL) {
					wrapper->mVertexList[source] = new Vertex<PTYPE>();
					wrapper->mVertexList[source]->oindex = source;
				}
				if (wrapper->mVertexList[target] == NULL) {
					wrapper->mVertexList[target] = new Vertex<PTYPE>();
					wrapper->mVertexList[target]->oindex = target;
				}
				new Edge<PTYPE>(wrapper->mVertexList[source], wrapper->mVertexList[target], prob);
			}
		}

		assert(wrapper->mInitialNodeID >= 0);

		assert(wrapper->mVertexList[wrapper->mInitialNodeID] != NULL);
		assert(wrapper->mVertexList[wrapper->mInitialNodeID]->oindex == (unsigned int ) wrapper->mInitialNodeID);

		LOG4CPLUS_TRACE(logger, "Initial node is " << wrapper->mInitialNodeID);

		if (DEFAULT_LOG_LEVEL >= TRACE_LOG_LEVEL) {
			stringstream ss;
			ss << "The following nodes are target nodes: ";
			for (unsigned int i = 0; i < wrapper->mTargetNodes.size(); i++) {
				if (wrapper->isTarget(i)) {
					ss << i << ", ";
				}
			}

			LOG4CPLUS_DEBUG(logger, ss.str());
		}

		return true;

		Error:

		return false;
	}
private:
	IOWrapper *wrapper;
};
}
#endif /* DTMCREADER_H_ */
