/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "PrismReader.h"
#include "../output/OutputFormatters.h"
#include "../param/parametric/ExprToNumber.h"
#include "../param/prismparser/Property.h"
#include "../param/rationalFunction/RationalFunction.h"
#include "../param/parametric/PMM.h"

namespace parametric {
class PMM;
}

using namespace rational;

namespace param_sccmc_io {
Logger prismlogger = Logger::getInstance("PrismReader");

/**
 * Constructor
 * @param wrapper wrapper for initialization of graph
 * @return
 */
PrismReader::PrismReader(IOWrapper *wrapper) {
	prismlogger.setLogLevel(DEFAULT_LOG_LEVEL);
	this->wrapper = wrapper;
	explorer = new Model2XExplorer(wrapper);
	exprToNumber = new ExprToNumber();
	explorer->setExprToNumber(*exprToNumber);
}

/**
 * Destructor
 * @return
 */
PrismReader::~PrismReader() {
	delete explorer;
	delete exprToNumber;
}

/**
 * Parses PRISM file and initializes graph in IOWrapper
 * @param modelFile .pm File
 * @param propertyFile .pctl File
 * @return true, if successful, false, if error occurred
 */
bool PrismReader::parse(string modelFile, string propertyFile) {
	explorer->setModelFilename(modelFile);
	explorer->setPropertyFilename(propertyFile);
	explorer->explore();

	//Set Targets
	if (explorer->getNumProperties() > 1) {
		LOG4CPLUS_ERROR(prismlogger, "We only consider first property!");
		return false;
	}
	ResultTarget result;
	for (unsigned int i = 0; i < wrapper->mVertexList.size(); i++) {
		result.push_back(false);
	}
	bool successfull = setTargets(explorer->getProperty(0), result);
	if (!successfull) {
		return false;
	}

	//New target
	for (unsigned int state = 0; state < result.size(); state++) {
		if (result[state]) {
			LOG4CPLUS_TRACE(prismlogger, "Target: " << state);
			wrapper->setTarget(state);

			//Remove outgoing edges
			unsigned int size = wrapper->mVertexList[state]->outEdges.size();
			for (int i = size - 1; i >= 0; i--) {
				Edge<PTYPE> *tmp = wrapper->mVertexList[state]->outEdges[i];
				tmp->target->inEdges.remove(tmp);
				wrapper->mVertexList[state]->outEdges.pop_back();
			}
		}
	}
	return true;
}

/**
 * Set target states from property
 * @param property property to check
 * @param result contains target states
 * @return true, if successful
 */
bool PrismReader::setTargets(Property property, ResultTarget& result) {
	LOG4CPLUS_TRACE(prismlogger, "Property: " << property);

	switch (property.getType()) {
	case ExprProp:
		LOG4CPLUS_TRACE(prismlogger, "Expression");
		setTargetsFromExpression(property, result);
		break;
	case AndProp:
	case OrProp:
	case ImplProp:
		LOG4CPLUS_TRACE(prismlogger, "Binary");
		setTargetsFromBinary(property, result);
		break;
	case NegProp:
		LOG4CPLUS_TRACE(prismlogger, "Neg");
		setTargetsFromNeg(property, result);
		break;
	case QuantProp:
		LOG4CPLUS_WARN(prismlogger, "Quant: Ignored at the moment");
		setTargets(property[0], result);
		break;
	case NextProp:
		LOG4CPLUS_WARN(prismlogger, "Next: Ignored at the moment");
		setTargets(property[0], result);
		break;
	case UntilProp:
		LOG4CPLUS_WARN(prismlogger, "Until: Ignored at the moment");
		setTargets(property[1], result);
		break;
	case ReachRewProp:
		LOG4CPLUS_WARN(prismlogger, "ReachReward: Ignored at the moment");
		setTargets(property[0], result);
		break;
	case SteadySProp:
		LOG4CPLUS_WARN(prismlogger, "Steady state: Ignored at the moment");
		setTargets(property[0], result);
		break;
	default:
		LOG4CPLUS_ERROR(prismlogger, "Cannot handle this property!");
		return false;
	}
	return true;
}

/**
 * Get targets from expression
 * @param property property which is expression
 * @param result targets
 */
void PrismReader::setTargetsFromExpression(Property property, ResultTarget& result) {
	unsigned int ap = exprToNumber->getNumberByExpr(property.getExpr());
	PMM* pmm = explorer->getPMMModel();
	for (PMM::state state(0); state < pmm->getNumStates(); state++) {
		result[state] = pmm->isAP(state, ap);
	}
}

/**
 * Get targets from binary property
 * @param property property which is binary
 * @param result targets
 */
void PrismReader::setTargetsFromBinary(Property property, ResultTarget& result) {
	setTargets(property[0], result);
	ResultTarget tmp2Result;
	for (unsigned int i = 0; i < result.size(); i++) {
		tmp2Result.push_back(false);
	}
	setTargets(property[1], tmp2Result);

	for (unsigned int i = 0; i < result.size(); i++) {
		if (OrProp == property.getType()) {
			result[i] = result[i] || tmp2Result[i];
		} else if (AndProp == property.getType()) {
			result[i] = result[i] && tmp2Result[i];
		} else if (ImplProp == property.getType()) {
			result[i] = !result[i] || tmp2Result[i];
		}
	}
}

/**
 * Get targets from negation
 * @param property property which is negation
 * @param result targets
 */
void PrismReader::setTargetsFromNeg(Property property, ResultTarget& result) {
	setTargets(property[0], result);

	for (unsigned int i = 0; i < result.size(); i++) {
		result[i] = !result[i];
	}
}
}
