/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef PROBREADER_H_
#define PROBREADER_H_

#include "../../defines.h"

#ifdef USE_EXACT
#include <gmpxx.h>
#endif
#ifdef USE_PARAMETRIC
#include "../../data/Parameters.h"
#include <ginac/ginac.h>
using namespace parametric;
#endif

using namespace std;

namespace param_sccmc_io {

template<class T>
class AbstractProbReader {
public:
	virtual T readProb(const string readValue) const {
		return 0;
	}

	virtual ~AbstractProbReader() {

	}
};

/*
 * "Default Default" Parser, relying solely on the parsing mechanism
 * of the >> operator. This suffices for the non-parametric-double case
 */
template<class T>
class DefaultProbReader: public AbstractProbReader<T> {
public:
	T readProb(const string readValue) const {
		T ret;
		istringstream stm;
		stm.str(readValue);
		stm >> ret;
		assert(ret > 0);
		return ret;
	}
};

/*
 * Default implementation for non-parametric-exact-arithmetic probabilities
 */
#ifdef USE_EXACT
template <>
class DefaultProbReader <mpq_class>: public AbstractProbReader<mpq_class> {
public:
	mpq_class readProb(const string readValue) const {
		mpq_class prob;
		if (readValue[0] == '1')
		prob = 1;
		else {
			//Prob is of form 0.x -> convert to fraction so that it can be parsed
			string nom = readValue.substr(2);
			string denom = "1";
			denom.append(nom.length(), '0');
			while (nom[0] == '0') nom = nom.substr(1);
			prob = nom + '/' + denom;

			//common factors are not cancelled automatically!
			prob.canonicalize();
		}
		assert(prob > 0);
		return prob;
	}
};
#endif

/*
 * Default implementation for parametric probabilities
 */
#ifdef USE_PARAMETRIC
template<>
class DefaultProbReader<Rational> : public AbstractProbReader<Rational> {
public:
	Rational readProb(const string readValue) const {
		parser reader(Parameters::getInstance().getParameterTable());
		reader.strict = true;
		ex e;
		try {
			e = reader(readValue);
		} catch (parse_error& err) {
			cerr << err.what() << " in " << readValue << endl;
		}
		e = e.numer_denom();
		assert(e.nops() == 2);

		Rational prob;
		if (is_a<numeric>(e.op(0))) {
			//Try to compose rational number from floating point input
			stringstream sStream;
			sStream << e.op(0);
			string s = sStream.str();
			size_t pos = s.find('.');
			if(pos != string::npos) {
				unsigned numDecDigits = s.size()-pos-1;
				string sNum = s.substr(0, pos) + s.substr(pos+1, numDecDigits);
				numeric numerator = numeric(sNum.c_str());

				string sDenom = "1" + string(numDecDigits, '0');
				numeric denominator = numeric(sDenom.c_str());
				denominator *= ex_to<numeric>(e.op(1));

				assert(is_a<numeric>(e.op(1)));
				assert(numerator.is_integer());
				assert(denominator.is_integer());
				assert( (e.op(0)/e.op(1)).expand() == (numerator/denominator).expand());

				if (numerator / denominator == 1) {
					prob = Parameters::getInstance().getOne();
				} else {
					prob = Rational(numerator, denominator);
				}
			} else {
				prob = Rational(e.op(0), e.op(1));
			}
		} else {
			prob = Rational(e.op(0), e.op(1));
		}
		Parameters::getInstance().checkConstraintOneZero(prob);
		return prob;
	}
};

#endif

}

#endif /* PROBREADER_H_ */
