/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "XMLReader.h"
#include "../output/OutputFormatters.h"

using namespace scc_cex;
using namespace std;

namespace param_sccmc_io {
Logger xmllogger = Logger::getInstance("XMLReader");

XMLReader::XMLReader(IOWrapper *wrapper) {
	xmllogger.setLogLevel(DEFAULT_LOG_LEVEL);
	this->wrapper = wrapper;
}

void XMLReader::readSCCInXML(Graph<PTYPE>* res, pugi::xml_node node) const {
	//pugi::xml_attribute sccId = node.attribute("id");
	//res->graph_id = sccId.as_int();

	//unsigned int value;

	pugi::xml_node vtx = node.child("vtx");
	pugi::xml_node inp = node.child("inp");
	pugi::xml_node out = node.child("out");
	pugi::xml_node edg = node.child("edg");

	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		if (wrapper->mVertexList[nodeId] != NULL) {
			wrapper->mVertexList[nodeId]->g = res;
		}
		res->inputVertices.push_back(wrapper->mVertexList[nodeId]);
		res->allVertices.push_back(wrapper->mVertexList[nodeId]);
		inp = inp.next_sibling("inp");
	}

	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		res->outputVertices.push_back(wrapper->mVertexList[nodeId]);
		out = out.next_sibling("out");
	}

	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		if (wrapper->mVertexList[nodeId] != NULL) {
			wrapper->mVertexList[nodeId]->g = res;
		}
		res->allVertices.push_back(wrapper->mVertexList[nodeId]);
		vtx = vtx.next_sibling("vtx");
	}

	param_sccmc_io::DefaultProbReader<PTYPE> probParser;
	Edge<PTYPE> *tempedge;
	while (edg) {
		bool abstract1 = edg.attribute("abs").as_bool();
		PTYPE prob = probParser.readProb(edg.attribute("prb").value());
		//PTYPE prob = probParser.parseProb(read_prob);
		unsigned int source = edg.attribute("src").as_uint();
		unsigned int target = edg.attribute("trg").as_uint();
		if (abstract1) {
			ConfigurationSingleton::getInstance()->setIoflCexConcrete(false);
		}

		tempedge = new Edge<PTYPE>(wrapper->mVertexList[source], wrapper->mVertexList[target], prob);
		tempedge->state = abstract1;
		tempedge->not_m_min = true;

		wrapper->mEdgeCount++;

		edg = edg.next_sibling("edg");
	}

	res->state = CONCRETE;
	res->concretized = true;

	//check if the scc is abstract, i.e. has at least one outgoing abstract edge
	for (vtx_it it = res->allVertices.begin(); it != res->allVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == ABSTRACT) {
				res->state = ABSTRACT;
				res->concretized = false;
			}
		}
	}

	for (vtx_it it = res->inputVertices.begin(); it != res->inputVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == ABSTRACT) {
				res->state = ABSTRACT;
				res->concretized = false;
			}
		}
	}

	int counter = 0;
	pugi::xml_node childScc = node.child("scc");
	while (childScc) {
		counter++;
		//pugi::xml_attribute sccId = childScc.attribute("id");
		//Graph<PTYPE> *temp = res->addSubgraph(sccId.as_int());
		Graph<PTYPE> *temp = res->addSubgraph();
		readSCCInXML(temp, childScc);
		//res->addSubgraph(temp);
		childScc = childScc.next_sibling("scc");
	}

	if (counter == 0) {
		//res->concretized = true;
	}

}

Graph<PTYPE>* XMLReader::getGraphFromXML(string xmlString) const {
	stringstream ss;
	ss << xmlString;
	return getGraphFromXML(ss);
}

Graph<PTYPE>* XMLReader::getGraphFromXML(istream& istream) const {
	stringstream stream;
	stream << istream.rdbuf();

	pugi::xml_document doc;
	//pugi::xml_parse_result result = doc.load_file(xmlPath.c_str());
	pugi::xml_parse_result result = doc.load(stream);
	LOG4CPLUS_INFO(xmllogger, "Load result: " << result.description());

	ConfigurationSingleton::getInstance()->setIoflCexConcrete(true);

	Graph<PTYPE> *resG = NULL; //new Graph<PTYPE>();

	const char* mcName;
	if (doc.child("dtmc")) {
		mcName = "dtmc";
	} else if (doc.child("pmc")) {
		mcName = "pmc";
	} else {
		LOG4CPLUS_ERROR(xmllogger, "Unexpected xml tag: " << doc.first_child());
		return NULL;
	}

	/* The first scc child contains all nodes, so we initialize the nodelist */

	pugi::xml_node firstScc = doc.child(mcName).child("scc");

	pugi::xml_node vtx = firstScc.child("vtx");
	pugi::xml_node out = firstScc.child("out");
	pugi::xml_node edg = firstScc.child("edg");
	pugi::xml_node inp = firstScc.child("inp");

	//read all vertices
	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		if (wrapper->mVertexList[nodeId] == NULL) {
			wrapper->mVertexList[nodeId] = new Vertex<PTYPE>();
			wrapper->mVertexList[nodeId]->oindex = nodeId;
		}
		vtx = vtx.next_sibling("vtx");
	}
	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		if (wrapper->mVertexList[nodeId] == NULL) {
			wrapper->mVertexList[nodeId] = new Vertex<PTYPE>();
			wrapper->mVertexList[nodeId]->oindex = nodeId;
		}
		inp = inp.next_sibling("out");
	}
	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		for (unsigned int i = wrapper->mVertexList.size(); i <= nodeId; i++) {
			wrapper->mVertexList.push_back(NULL);
		}
		if (wrapper->mVertexList[nodeId] == NULL) {
			wrapper->mVertexList[nodeId] = new Vertex<PTYPE>();
			wrapper->mVertexList[nodeId]->oindex = nodeId;
		}
		out = out.next_sibling("out");
	}

	//all vertices have been read, since input states also belong to the allVertices

	//read the target states
	pugi::xml_node tar = doc.child(mcName).child("targets").child("target");
	while (tar) {
		unsigned int nodeId = atoi(tar.child_value());
		wrapper->setTarget(nodeId);
		tar = tar.next_sibling("target");
	}

#ifdef USE_PARAMETRIC
	//read the parameters
	pugi::xml_node param = doc.child(mcName).child("parameters").child("parameter");
	while (param) {
		parametric::Parameters::getInstance().addVariable(param.child_value());
		param = param.next_sibling("parameter");
	}
#endif

	inp = firstScc.child("inp");
	if (inp) {
		unsigned int id = atoi(inp.child_value());
		wrapper->mInitialNode = wrapper->mVertexList[id];
		wrapper->mInitialNodeID = id;
		if (inp.next_sibling("inp")) {
			LOG4CPLUS_WARN(xmllogger, "Only node " << id << " is handled as input node, other input nodes are ignored");
		}
	} else {
		LOG4CPLUS_WARN(xmllogger, "No input node specified");
	}

	DefaultProbReader<PTYPE> probParser;
	Edge<PTYPE> *tempedge;
	while (edg) {
		bool abstract = edg.attribute("abs").as_bool();
		PTYPE prob = probParser.readProb(edg.attribute("prb").value());
		unsigned int source = edg.attribute("src").as_uint();
		unsigned int target = edg.attribute("trg").as_uint();
		if (abstract) {
			ConfigurationSingleton::getInstance()->setIoflCexConcrete(false);
		}
		if (!wrapper->isTarget(source)) {
			tempedge = new Edge<PTYPE>(wrapper->mVertexList[source], wrapper->mVertexList[target], prob);
			tempedge->state = abstract;
			tempedge->not_m_min = true;

			wrapper->mEdgeCount++;
		}

		edg = edg.next_sibling("edg");
	}

	if (resG == NULL) {
		resG = new Graph<PTYPE>();
		//pugi::xml_attribute sccId = firstScc.attribute("id");
		//resG->graph_id = sccId.as_int();
	}

	out = firstScc.child("out");
	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		//		for (int i = mVertexList.size(); i <= nodeId; i++) {
		//			mVertexList.push_back(NULL);
		//		}
		if (wrapper->mVertexList[nodeId] != NULL) {
			//mVertexList[nodeId] = new Vertex<PTYPE>();
			wrapper->mVertexList[nodeId]->oindex = nodeId;
			//mVertexList[nodeId]->g = NULL;
		}
		resG->outputVertices.push_back(wrapper->mVertexList[nodeId]);
		out = out.next_sibling("out");
	}

	vtx = firstScc.child("vtx");
	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		//		for (int i = mVertexList.size(); i <= nodeId; i++) {
		//			mVertexList.push_back(NULL);
		//		}
		if (wrapper->mVertexList[nodeId] != NULL) {
			//mVertexList[nodeId] = new Vertex<PTYPE>();
			wrapper->mVertexList[nodeId]->oindex = nodeId;
			wrapper->mVertexList[nodeId]->g = resG;
		}
		resG->allVertices.push_back(wrapper->mVertexList[nodeId]);
		vtx = vtx.next_sibling("vtx");
	}

	inp = firstScc.child("inp");
	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		if (wrapper->mVertexList[nodeId] != NULL) {
			resG->inputVertices.push_back(wrapper->mVertexList[nodeId]);
			wrapper->mVertexList[nodeId]->g = resG;
		}
		inp = inp.next_sibling("inp");
		resG->allVertices.push_back(wrapper->mVertexList[nodeId]);
	}

	pugi::xml_node childScc = firstScc.child("scc");

	//the most inner scc, i.e which have no child scc are the conretized ones
	if (!childScc) {
		resG->concretized = true;
	}

	resG->state = CONCRETE;
	resG->concretized = true;

	//check if the scc is abstract, i.e. has at least one outgoing abstract edge
	for (vtx_it it = resG->allVertices.begin(); it != resG->allVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == true) {
				resG->state = ABSTRACT;
				resG->concretized = false;
			}
		}
	}

	for (vtx_it it = resG->inputVertices.begin(); it != resG->inputVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == true) {
				resG->state = ABSTRACT;
				resG->concretized = false;
			}
		}
	}

	while (childScc) {
		//pugi::xml_attribute sccId = childScc.attribute("id");
		//Graph<PTYPE> *temp = resG->addSubgraph(sccId.as_int());
		Graph<PTYPE> *temp = resG->addSubgraph();
		readSCCInXML(temp, childScc);
		childScc = childScc.next_sibling("scc");
	}

	AbstractOutputFormatter<PTYPE> *formatter;
	formatter = new XMLOutputFormatter<PTYPE>(wrapper, true, false);
	LOG4CPLUS_INFO(xmllogger, formatter->generateOutput(*resG));

	return resG;

}
}
