/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef OUTPUTFORMATTERS_H_
#define OUTPUTFORMATTERS_H_

#include "../../data/GraphTypes.h"
#include "../IOWrapper.h"
#include <string>

#ifdef USE_PARAMETRIC
#include "../../data/Parameters.h"
#endif

#include <iostream>
#include <sstream>

using namespace std;

namespace scc_cex {

template<class T>
class AbstractOutputFormatter {
public:
	virtual string generateOutput(const Graph<T>& g) {
		return "__default__";
	}

	virtual ~AbstractOutputFormatter() {

	}
};

template<class T>
class DefaultOutputFormatter: public AbstractOutputFormatter<T> {
public:

	DefaultOutputFormatter(Vertex<T> *initNode, bool keepAbstract) {
		mInitNode = initNode;
		mKeepAbstract = keepAbstract;
	}

	DefaultOutputFormatter(Vertex<T> *initNode, std::vector<Vertex<T>*> targetNodes) {
		mInitNode = initNode;
		mTargetNodes = targetNodes;
		mKeepAbstract = true;
	}

	string generateOutput(const Graph<T>& g);
	void recursiveEdgeOutput(stringstream *oss, Graph<T> *g, vector<Edge<PTYPE> > *eList);
	bool edgeInList(vector<Edge<PTYPE> > *eList, Edge<T> *el);
private:
	Vertex<T> *mInitNode;
	std::vector<Vertex<T>*> mTargetNodes;
	bool mKeepAbstract;
	std::vector<int> mStates;
};

template<class T>
class XMLOutputFormatter: public AbstractOutputFormatter<T> {
public:
	XMLOutputFormatter(Vertex<T> *initNode, std::vector<bool> targetNodes, bool keepAbstract, bool concreteSearch) :
			mInitNode(initNode), mTargetNodes(targetNodes), mKeepAbstract(keepAbstract), mConcreteSearch(concreteSearch) {
	}

	XMLOutputFormatter(const IOWrapper *wrapper, bool keepAbstract, bool concreteSearch) :
			mInitNode(wrapper->mInitialNode), mTargetNodes(wrapper->mTargetNodes), mKeepAbstract(keepAbstract), mConcreteSearch(
					concreteSearch) {
	}

	string generateOutput(const Graph<T>& g);
	void recursiveTransformation(const Graph<T>& g, stringstream& sSCC, const int depth);
private:
	string edgeTag(const Edge<T>* e) const;
	string edgeTag(const Edge<T>* e, const char type, const Graph<T>& g) const; //The debugging version
	//Some methods used for debugging purposes
	void markEdgesUnseen(vector<Vertex<PTYPE>*> vertices);
	void
	printUnseenEdges(const vector<Vertex<PTYPE>*> vertices, stringstream& sSCC, const char type, const Graph<PTYPE>& g) const;
	bool edgeInXML(bool graphConcretized, const Edge<T>* edge) const;
	Vertex<T> *mInitNode;
	std::vector<bool> mTargetNodes;
	bool mKeepAbstract;
	bool mConcreteSearch;
};

/*
 * #############################################################################
 * ############################# DefaultOutputFormatter ########################
 * #############################################################################
 */

template<class T>
string DefaultOutputFormatter<T>::generateOutput(const Graph<T> & g) {
	stringstream ss;

	//read all edges

	vector<Edge<PTYPE> > allEdgesList;
	vector<Vertex<PTYPE>*> vertices = g.allVertices;
	for (unsigned int i = 0; i < vertices.size(); i++) {
		Vertex<PTYPE>* v = vertices[i];
		vector<Edge<PTYPE>*> edgeList;
		for (unsigned int j = 0; j < v->outEdges.size(); j++) {
			Edge<PTYPE>* e = v->outEdges[j];

			if (!mKeepAbstract) {
				if (e->state)
					continue;
			}
			if (!edgeInList(&allEdgesList, e)) {
				if (find(mStates.begin(), mStates.end(), e->target->oindex) == mStates.end())
					mStates.push_back(e->target->oindex);
				if (find(mStates.begin(), mStates.end(), e->source->oindex) == mStates.end())
					mStates.push_back(e->source->oindex);
				allEdgesList.push_back(*e);
			}
		}
	}
	for (unsigned int j = 0; j < g.subgraphs.size(); j++) {
		recursiveEdgeOutput(&ss, g.subgraphs[j], &allEdgesList);
	}

	sort(allEdgesList.begin(), allEdgesList.end());

	ss << "STATES " << mStates.size() << endl;
	ss << "TRANSITIONS " << allEdgesList.size() << endl;

	//Initial node
	ss << "INITIAL ";
	if (mInitNode != NULL) {
		assert(mInitNode->outEdges.size() >= 1);
		for (unsigned int i = 0; i < mInitNode->outEdges.size(); i++) {
			if (mInitNode->outEdges[i]->state == CONCRETE) {
				ss << mInitNode->outEdges[0]->target->oindex;
				break;
			}
			assert(i < mInitNode->outEdges.size());
		}
	} else {
		assert(g.inputVertices.size() == 1);
		ss << g.inputVertices[0]->oindex;
	}
	ss << endl;

	//Target nodes
	if (mTargetNodes.empty()) {
		for (unsigned int i = 0; i < g.outputVertices.size(); i++)
			ss << "TARGET " << g.outputVertices[i]->oindex << endl;
	} else {
		for (unsigned int i = 0; i < mTargetNodes.size(); i++)
			ss << "TARGET " << mTargetNodes.at(i)->oindex << endl;
	}

#ifdef USE_PARAMETRIC
	//Parameters
	symtab parameters = parametric::Parameters::getInstance().getParameterTable();
	for (symtab::iterator iter = parameters.begin(); iter != parameters.end(); iter++) {
		ss << "PARAMETER " << iter->second << endl;
	}
#endif

	//Edges
	for (vector<Edge<PTYPE> >::iterator it = allEdgesList.begin(); it != allEdgesList.end(); ++it) {
		ss << (*it).source->oindex << " " << (*it).target->oindex << " " << (*it).prob << endl;
	}

	assert(g.inputVertices.size() == 1);
	return ss.str();
}

template<class T>
bool DefaultOutputFormatter<T>::edgeInList(vector<Edge<PTYPE> > *eList, Edge<T> *el) {
	for (unsigned int i = 0; i < eList->size(); i++) {
		if ((*eList)[i].source->oindex != el->source->oindex) {
			continue;
		} else if ((*eList)[i].target->oindex != el->target->oindex) {
			continue;
		} else {
			return true;
		}
	}
	return false;
}

template<class T>
void DefaultOutputFormatter<T>::recursiveEdgeOutput(stringstream* oss, Graph<T> *g, vector<Edge<PTYPE> > *eList) {
	vector<Vertex<PTYPE>*> vertices = g->allVertices;
	for (unsigned int i = 0; i < vertices.size(); i++) {
		Vertex<PTYPE>* v = vertices[i];
		vector<Edge<PTYPE>*> edgeList;
		for (unsigned int j = 0; j < v->outEdges.size(); j++) {
			Edge<PTYPE>* e = v->outEdges[j];

			if (!mKeepAbstract) {
				if (e->state)
					continue;
			}
			if (!edgeInList(eList, e)) {
				if (find(mStates.begin(), mStates.end(), e->target->oindex) == mStates.end())
					mStates.push_back(e->target->oindex);
				if (find(mStates.begin(), mStates.end(), e->source->oindex) == mStates.end())
					mStates.push_back(e->source->oindex);
				eList->push_back(*e);
			}
		}
	}
	for (unsigned int j = 0; j < g->subgraphs.size(); j++) {
		recursiveEdgeOutput(oss, g->subgraphs[j], eList);
	}
}

/*
 * #############################################################################
 * ################################# XMLOutputFormatter ########################
 * #############################################################################
 */

/*
 * Returns a line with an edge tag representing the given edge e.
 */
template<class T>
string XMLOutputFormatter<T>::edgeTag(const Edge<T>* e, const char type, const Graph<T>& g) const {
	stringstream ss;
	ss << "<edg abs=\"" << e->state << "\" prb=\"" << e->prob << "\" src=\"" << e->source->oindex << "\" trg=\""
			<< e->target->oindex
			//This info might be useful for debugging / DEBUGGING purposes,
			//but does not contain any information necessary to reconstruct the graph
			<< "\" type=\"" << type
#ifdef DEBUGGING
			<< "\" grs=\"" << (e->source->g != NULL ? e->source->g->graph_id : -1)
			<< "\" grt=\"" << (e->target->g != NULL ? e->target->g->graph_id : -1)
			<< "\" caller=\"" << g.graph_id
#endif
			<< "\" />\n";
	return ss.str();
}

template<class T>
string XMLOutputFormatter<T>::edgeTag(const Edge<T>* e) const {
	stringstream ss;
	ss << "<edg abs=\"" << e->state << "\" prb=\"" << e->prob << "\" src=\"" << e->source->oindex << "\" trg=\""
			<< e->target->oindex << "\" />\n";
	return ss.str();
}

/*
 * Traverses the graph structure g recursively and maps it to the XML format
 * All tags are appended to the stringstream sSCC
 * depth is used to keep track of the indentation - if indentation is desired
 * (see #define at the beginning of this file)
 */
template<class T>
void XMLOutputFormatter<T>::recursiveTransformation(const Graph<T>& g, stringstream& sSCC, const int depth) {
	string indent = "";
	indent.append(depth << 1, ' ');

	if (mKeepAbstract || (!mConcreteSearch && !g.concretized) || depth == 1) {
		//Print only for root or abstract SCCs, if desired
		sSCC << indent << "<scc";
		sSCC << " id=\"" << g.graph_id << "\"";
#ifdef DEBUGGING
		sSCC << " state=\"" << g.state << "\"";
		sSCC << " conc=\"" << g.concretized << "\"";
		for (unsigned int i = 0; i < g.inputVertices.size(); i++) {
			if (mKeepAbstract || g.inputVertices[i]->isInClosure) {
				sSCC << " node" << i << "=\"" << g.inputVertices[i]->oindex << "\"";
			}
		}
#endif
		sSCC << ">\n";

		//(1) Vertices
		for (unsigned int i = 0; i < g.inputVertices.size(); i++) {
			if (mKeepAbstract || g.inputVertices[i]->inXML) {
				sSCC << indent << "  " << "<inp gid=\"" << g.inputVertices[i]->g->graph_id << "\">"
						<< g.inputVertices[i]->oindex << "</inp>\n";
			}
		}

		for (unsigned int i = 0; i < g.outputVertices.size(); i++) {
			if (mKeepAbstract || g.outputVertices[i]->inXML) {
				int graph_id = -1;
				if (g.outputVertices[i]->g) {
					graph_id = g.outputVertices[i]->g->graph_id;
				}
				sSCC << indent << "  " << "<out gid=\"" << graph_id << "\">" << g.outputVertices[i]->oindex << "</out>\n";
			}
		}

		for (unsigned int i = 0; i < g.allVertices.size(); i++) {
			if (mKeepAbstract || g.allVertices[i]->inXML) {
				//Check if vertex is input
				bool input = false;

				unsigned int id = g.allVertices[i]->oindex;
				for (unsigned int j = 0; j < g.inputVertices.size(); j++) {
					if (g.inputVertices[j]->oindex == id) {
						input = true;
						break;
					}
				}
				if (!input) {
					// vertex is no input
					sSCC << indent << "  " << "<vtx gid=\"" << g.allVertices[i]->g->graph_id << "\">" << id << "</vtx>\n";
				}

			}
		}
	}

	//(2) Subgraphs
	for (unsigned int i = 0; i < g.subgraphs.size(); i++) {
		if (mKeepAbstract || g.subgraphs[i]->inXML) {
			recursiveTransformation(*(g.subgraphs[i]), sSCC, depth + 1);
		}
	}

	//(3) Edges
	//Start with the input edges
	for (unsigned int i = 0; i < g.inputVertices.size(); i++) {
		for (unsigned int j = 0; j < g.inputVertices[i]->outEdges.size(); j++) {
			Edge<T>* e = g.inputVertices[i]->outEdges[j];
			if (edgeInXML(g.concretized, e)) {
#ifdef DEBUGGING
				sSCC << indent << "  " << edgeTag(e, 'I', g);
#else
				sSCC << indent << "  " << edgeTag(e);
#endif
				e->setXMLMark(true);
			}
		}
	}

	//Continue with the inner edges
	for (unsigned int i = 0; i < g.allVertices.size(); i++) {
		//Check if this vertex belongs to g
		if (g.allVertices[i]->g == (&g)) {
			for (unsigned int j = 0; j < g.allVertices[i]->inEdges.size(); j++) {
				Edge<T>* e = g.allVertices[i]->inEdges[j];
				if (!e->getXMLMark()) {
					if (e->source->g == (&g)) {
						if (edgeInXML(g.concretized, e)) {
#ifdef DEBUGGING
							sSCC << indent << "  " << edgeTag(e, 'A', g);
#else
							sSCC << indent << "  " << edgeTag(e);
#endif

							e->setXMLMark(true);
						}
					}
				}
			}

			for (unsigned int j = 0; j < g.allVertices[i]->outEdges.size(); j++) {
				Edge<T>* e = g.allVertices[i]->outEdges[j];
				if (!e->getXMLMark()) {
					if (e->target->g == (&g) || e->target->isInputVertex) {
						if (edgeInXML(g.concretized, e)) {
#ifdef DEBUGGING
							sSCC << indent << "  " << edgeTag(e, 'A', g);
#else
							sSCC << indent << "  " << edgeTag(e);
#endif
							e->setXMLMark(true);
						}
					}
				}
			}
		}

	}

	//Conclude with the outgoing edges
	for (unsigned int i = 0; i < g.outputVertices.size(); i++) {
		for (unsigned int j = 0; j < g.outputVertices[i]->inEdges.size(); j++) {
			Edge<T>* e = g.outputVertices[i]->inEdges[j];
			if (!e->getXMLMark()) {
				if (e->source->g == (&g)) {
					if (edgeInXML(g.concretized, e)) {
#ifdef DEBUGGING
						sSCC << indent << "  " << edgeTag(e, 'O', g);
#else
						sSCC << indent << "  " << edgeTag(e);
#endif
						e->setXMLMark(true);
					}
				}
			}
		}
	}

	if (mKeepAbstract || (!mConcreteSearch && !g.concretized) || depth == 1) {
		sSCC << indent << "</scc>\n";
	}
}

template<class T>
bool XMLOutputFormatter<T>::edgeInXML(bool graphConcretized, const Edge<T>* edge) const {
	if (mKeepAbstract) {
		return true;
	} else if (mConcreteSearch) {
		if (edge->isAbstract()) {
			return false;
		} else {
			return !edge->not_m_min;
		}
	} else if (!graphConcretized && edge->source->inXML && edge->target->inXML) {
		return true;
	} else if (edge->getMark()) {
		return true;
	} else
		return false;
}

template<class T>
void XMLOutputFormatter<T>::printUnseenEdges(const vector<Vertex<PTYPE>*> vertices, stringstream& sSCC, const char type,
		const Graph<PTYPE>& g) const {
	for (unsigned int i = 0; i < vertices.size(); i++) {
		for (unsigned int j = 0; j < vertices[i]->inEdges.size(); j++) {
			Edge<PTYPE>* e = vertices[i]->inEdges[j];
			if (!e->getXMLMark()) {
				sSCC << "    " << XMLOutputFormatter<PTYPE>::edgeTag(e, type, g);
				e->setXMLMark(true);
			}
		}
		for (unsigned int j = 0; j < vertices[i]->outEdges.size(); j++) {
			Edge<PTYPE>* e = vertices[i]->outEdges[j];
			if (!e->getXMLMark()) {
				sSCC << "    " << XMLOutputFormatter<PTYPE>::edgeTag(e, type, g);
				e->setXMLMark(true);
			}
		}
	}
}

// Map the graph to an XML structure
template<class T>
string XMLOutputFormatter<T>::generateOutput(const Graph<T>& g) {
	reinitializeEdgesForXML(g.allVertices);

	stringstream sSCC;

#ifdef USE_PARAMETRIC
	//Parameters
	symtab parameters = parametric::Parameters::getInstance().getParameterTable();
	if (!parameters.empty()) {
		sSCC << "  <parameters>\n";
		for (symtab::iterator iter = parameters.begin(); iter != parameters.end(); iter++) {
			sSCC << "    <parameter>" << iter->second << "</parameter>\n";
		}
		sSCC << "  </parameters>\n";
	}
#endif

	recursiveTransformation(g, sSCC, 1);

	if (DEFAULT_LOG_LEVEL == TRACE_LOG_LEVEL) {
		//Print all edges of all vertices of top level g
		sSCC << "  <unseenedges>\n";
		//markEdgesUnseen(g.allVertices);
		//markEdgesUnseen(g.inputVertices);
		//markEdgesUnseen(g.outputVertices);
		printUnseenEdges(g.allVertices, sSCC, 'A', g);
		printUnseenEdges(g.inputVertices, sSCC, 'I', g);
		printUnseenEdges(g.outputVertices, sSCC, 'O', g);
		sSCC << "  </unseenedges>\n";
	}

	//Set target states explicitly
	sSCC << "  <targets>\n";
	for (unsigned int i = 0; i < mTargetNodes.size(); i++) {
		if (mTargetNodes.at(i)) {
			sSCC << "    <target>" << i << "</target>\n";
		}
	}
	sSCC << "  </targets>\n";

#ifdef USE_PARAMETRIC
	if (!parameters.empty()) {
		//Print constraints
		sSCC << "  <constraints>\n";

		parametric::ConstraintSet constraints = parametric::Parameters::getInstance().getConstraints();
		for (parametric::constraint_const_it iter = constraints.begin(); iter != constraints.end(); iter++) {
			parametric::Constraint* constraint = *iter;
			sSCC << "    <constraint>" << constraint->toXMLString() << "</constraint>\n";
		}

		sSCC << "  </constraints>\n";

		return "<pmc>\n" + sSCC.str() + "</pmc>";
	}
#endif

	return "<dtmc>\n" + sSCC.str() + "</dtmc>";
}

template<class T>
void reinitializeEdgesForXML(std::vector<Vertex<T>*> vertices) {
	for (unsigned int i = 0; i < vertices.size(); i++) {
		for (edge_it it = vertices[i]->outEdges.begin(); it != vertices[i]->outEdges.end(); it++) {
			(*it)->setXMLMark(false);
		}
	}
}

}

#endif /* OUTPUTFORMATTERS_H_ */
