/*
 * This file is part of Model2X.
 *
 * Model2X is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Model2X is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Model2X. If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2010-2012 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#include "Model2X.h"
#include <cassert>
#include <limits>
#include <iostream>
#include <fstream>
#include <sstream>
extern "C" {
#include <luajit-2.0/lua.h>
#include <luajit-2.0/lauxlib.h>
#include <luajit-2.0/lualib.h>
}
#include "../rationalFunction/RationalFunction.h"
#include "../prismparser/Property.h"
#include "../prismparser/Model.h"
#include "../prismparser/PRISMParser.h"
#include "expr2lua.h"
#include "Model2XError.h"
#include "InitStatesComp.h"

namespace model2x {
using namespace std;
using namespace prismparser;
using namespace rational;

/**
 * Construct new Model-to-Lua converter.
 */
Model2X::Model2X() {
	model = NULL;
	succStatesList = NULL;
	succRatesList = NULL;
	succRewardsList = NULL;
	nonDetBounds = NULL;
	useRewards = false;
	initStates = NULL;
	exprNumbers = NULL;
	luaState = NULL;
	encDecSupport = NULL;
	rewardStruct = 0;
}

/**
 * Sets model to be explored.
 *
 * @param __model model to be used
 */
void Model2X::setModel(Model &__model) {
	model = &__model;
}

/**
 * Set whether a reward analysis shall be done.
 *
 * @param __useRewards true for reward analysis
 */
void Model2X::setUseRewards(bool __useRewards) {
	useRewards = __useRewards;
}

void Model2X::setRewardStruct(const unsigned __rewardStruct) {
	rewardStruct = __rewardStruct;
}

/**
 * Return number of states obtained so far.
 *
 * @return number of model states
 */
unsigned Model2X::getNumStates() const {
	assert(NULL != luaState);
	return numStates;
}

/**
 * Compute successors of a state @a state.
 * Successors will be written to successor list to be read later.
 *
 * @param state state to get successors of
 */
void Model2X::getStateSuccessors(unsigned state) {
	assert(NULL != luaState);
	lua_getglobal(luaState, "getStateSuccessors");
	decodeState(state);
	lua_call(luaState, 1, LUA_MULTRET);
	numSuccStates = lua_tointeger(luaState, 1);
	const unsigned luaAdd(1);
	const unsigned numStatesAdd(1);
	const unsigned rewardsAdd(1);
	const unsigned valSize(1);
	const unsigned succEntrySize(encStateSize + valSize + rewardsAdd);
	const unsigned nonDetBoundsSize(lua_tonumber(luaState, luaAdd + numStatesAdd + numSuccStates * succEntrySize));
	const unsigned nonDetSizeAdd = 1;
	for (unsigned boundNr(0); boundNr < nonDetBoundsSize; boundNr++) {
		nonDetBounds[boundNr] = lua_tonumber(luaState,
				luaAdd + numStatesAdd + numSuccStates * succEntrySize + nonDetSizeAdd + boundNr);
	}
	numNonDet = nonDetBoundsSize;
	RationalFunction succSum(0);
	RationalFunction reward = 0;
	unsigned lastNonDet(0);
	for (unsigned succNr = 0; succNr < numSuccStates; succNr++) {
		const unsigned state = insertState(luaAdd + numStatesAdd + succNr * succEntrySize);

		succStatesList[succNr] = state;
		const double rate(lua_tonumber(luaState, luaAdd + numStatesAdd + succNr * succEntrySize + encStateSize));
		succRatesList[succNr] = compRateValue(rate);
		if ((DTMC == model->getModelType()) || ((CTMC == model->getModelType()) && useRewards)) {
			succSum += succRatesList[succNr];
		}
		if (useRewards) {
			const double rewardPos(
					lua_tonumber(luaState, luaAdd + numStatesAdd + succNr * succEntrySize + encStateSize + valSize));
			RationalFunction rew;
			valueCompute.compute(succRewardValues[rewardPos], currState, rew);
			reward += succRatesList[succNr] * rew;
		}
		if (nonDetBounds[lastNonDet + 1] == succNr + 1) {
			if (useRewards) {
				RationalFunction stateReward;
				valueCompute.compute(stateRewardValue, currState, stateReward);
				if (DTMC != model->getModelType()) {
					reward += stateReward;
				}
				if (!isNonDet()) {
					reward /= succSum;
				}
				if (DTMC == model->getModelType()) {
					reward += stateReward;
				}
				succRewardsList[lastNonDet] = reward;
				reward = 0;
			}
			lastNonDet++;
		}
	}
	if (DTMC == model->getModelType()) {
		for (unsigned succ(0); succ < numSuccStates; succ++) {
			succRatesList[succ] /= succSum;
		}
	}
	lua_settop(luaState, 0);
	numStates = stateList.size() / encStateSize;
}

unsigned Model2X::encodeState(unsigned begin) {
	const unsigned startIndex(stateList.size());
	stateList.resize(stateList.size() + encStateSize, 0);
	unsigned bitindex = 0;
	for (unsigned varNr = 0; varNr < variablesInC.size(); varNr++) {
		unsigned varValue;
		if (encDecSupport[varNr] == numeric_limits<unsigned>::max()) {
			varValue = unsigned(lua_toboolean(luaState, begin + varNr));
			const unsigned resultIndex = startIndex + (bitindex / intBits);
			stateList[resultIndex] |= (varValue << (bitindex % intBits));
			bitindex++;
		} else {
			const unsigned varBits(encDecSupport[varNr]);
			varValue = unsigned(lua_tointeger(luaState, begin + varNr) - lowerBounds[varNr]);
			const unsigned resultIndex = startIndex + (bitindex / intBits);
			stateList[resultIndex] |= (varValue << (bitindex % intBits));
			bitindex += varBits;
		}
	}
	if (0 == stateMap.count(startIndex)) {
		stateMap.insert(make_pair(startIndex, startIndex));
		return startIndex / encStateSize;
	} else {
		const unsigned result(stateMap[startIndex] / encStateSize);
		stateList.resize(startIndex);
		return result;
	}
}

unsigned Model2X::insertState(unsigned begin) {
	const unsigned startIndex(stateList.size());
	stateList.resize(stateList.size() + encStateSize, 0);
	for (unsigned placeNr = 0; placeNr < encStateSize; placeNr++) {
		stateList[placeNr + startIndex] = unsigned(lua_tointeger(luaState, begin + placeNr));
	}

	if (0 == stateMap.count(startIndex)) {
		stateMap.insert(make_pair(startIndex, startIndex));
		return startIndex / encStateSize;
	} else {
		const unsigned result(stateMap[startIndex] / encStateSize);
		stateList.resize(startIndex);
		return result;
	}
}

void Model2X::printEncodeState(stringstream &stream) const {
	for (unsigned placeNr(0); placeNr < encStateSize; placeNr++) {
		stream << "      local place" << placeNr << " = 0\n";
	}
	unsigned bitindex = 0;
	for (unsigned varNr = 0; varNr < variablesInC.size(); varNr++) {
		if (encDecSupport[varNr] == numeric_limits<unsigned>::max()) {
			const unsigned resultIndex = bitindex / intBits;
			stream << "      if (" << variablesInC[varNr] << "P) then\n";
			stream << "        local shiftOne = bit.lshift(1, " << (bitindex % intBits) << ")\n";
			stream << "        place" << resultIndex << " = bit.bor(place" << resultIndex << ", shiftOne)\n";
			stream << "      end\n";
			bitindex++;
		} else {
			const unsigned varBits(encDecSupport[varNr]);
			stream << "      " << variablesInC[varNr] << "P = " << variablesInC[varNr] << "P - " << lowerBounds[varNr] << "\n";
			const unsigned resultIndex = bitindex / intBits;
			stream << "      place" << resultIndex << " = bit.bor(place" << resultIndex << ", bit.lshift("
					<< variablesInC[varNr] << "P, " << bitindex % intBits << "))\n";
			bitindex += varBits;
		}
	}
	for (unsigned placeNr(0); placeNr < encStateSize; placeNr++) {
		stream << "      table.insert(resultTable, place" << placeNr << ")\n";
	}
	stream << "      table.insert(resultTable, rate__)\n";
	stream << "      table.insert(resultTable, reward__)\n";
}

void Model2X::decodeState(unsigned encoded) {
	lua_newtable(luaState);
	const unsigned slBeginIndex(encoded * encStateSize);
	unsigned bitindex = 0;
	for (unsigned varNr = 0; varNr < variablesInC.size(); varNr++) {
		unsigned varBits(encDecSupport[varNr]);
		if (numeric_limits<unsigned>::max() == varBits) {
			varBits = 1;
		}
		const unsigned encIndex = slBeginIndex + (bitindex / intBits);
		const unsigned mask = numeric_limits<unsigned>::max() >> (intBits - varBits);
		unsigned varValue = (stateList[encIndex] >> (bitindex % intBits)) & mask;
		int sVarValue = (int) varValue + lowerBounds[varNr];
		if (numeric_limits<unsigned>::max() == encDecSupport[varNr]) {
			currState[varNr] = varValue;
			lua_pushstring(luaState, variables[varNr].toString().c_str());
			lua_pushboolean(luaState, varValue);
			lua_settable(luaState, -3);
			bitindex++;
		} else {
			currState[varNr] = sVarValue;
			lua_pushstring(luaState, variables[varNr].toString().c_str());
			lua_pushnumber(luaState, sVarValue);
			lua_settable(luaState, -3);
			bitindex += varBits;
		}
	}
}

/**
 * Return number of successors from last call of @a getStateSuccessors.
 *
 * @return number of successor states
 */
unsigned Model2X::getNumSuccStates() const {
	assert(NULL != luaState);
	return numSuccStates;
}

/**
 * Get array of successor states.
 * This value won't change for calls of @a getStateSuccessors. For this,
 * you need to call the function just once after calling @a build.
 *
 * @return array of successor states
 */
const unsigned *Model2X::getSuccStatesList() const {
	assert(NULL != luaState);
	return succStatesList;
}

/**
 * Get array of rates to successor states.
 * This value won't change for calls of @a getStateSuccessors.  For this,
 * you need to call the function just once after calling @a build.
 *
 * @return array of rates to successor states
 */
const ValueType *Model2X::getSuccRatesList() const {
	assert(NULL != luaState);
	return succRatesList;
}

/**
 * Get array of rewards to successor states.
 * This value won't change for calls of @a getStateSuccessors. For this,
 * you need to call the function just once after calling @a build.
 *
 * @return array of rates to successor states
 */
const ValueType *Model2X::getSuccRewardsList() const {
	assert(NULL != luaState);
	return succRewardsList;
}

/**
 * Get non-determinism bounds.
 * This value won't change for calls of @a getStateSuccessors. For this,
 * you need to call the function just once after calling @a build.
 * TODO more explanation
 *
 * @return array of bounds of non-determinism
 */
const unsigned *Model2X::getNonDetBounds() const {
	assert(NULL != luaState);
	return nonDetBounds;
}

/**
 * Check whether state is in given state set.
 *
 * @param state state to check whether it is in set
 * @param setNr number of set to check whether state is in
 * @return true iff given state is in given set
 */
bool Model2X::inStateSet(unsigned state, unsigned setNr) {
	lua_getglobal(luaState, "inStateSet");
	lua_pushnumber(luaState, setNr);
	decodeState(state);
	lua_call(luaState, 1 + 1, 1);
	const bool result = lua_toboolean(luaState, -1);
	lua_pop(luaState, 1);
	return result;
}

/**
 * Get number of initial states.
 *
 * @return number of initial states
 */
unsigned Model2X::getNumInitStates() const {
	return numInitStates;
}

/**
 * Get list of initial states.
 *
 * @return list of initial states
 */
const unsigned *Model2X::getInitStates() const {
	return initStates;
}

/**
 * Get number of non-deterministic choices.
 * For models without non-determinism, this value will always be 1.
 */
unsigned Model2X::getNumNonDet() const {
	return numNonDet;
}

void Model2X::prepareStateFormulasFromPropertyStructure(const Property &prop) {
	if (ExprProp == prop.getType()) {
		addStateSet(prop.getExpr());
		(*exprNumbers)[prop.getExpr()] = stateSets.size() - 1;
	} else {
		for (unsigned childNr(0); childNr < prop.arity(); childNr++) {
			prepareStateFormulasFromPropertyStructure(prop[childNr]);
		}
	}
}

void Model2X::prepareStateFormulasFromPropertyStructure() {
	for (unsigned propNr = 0; propNr < model->getNumProperties(); propNr++) {
		const Property &prop(model->getProperty(propNr));
		prepareStateFormulasFromPropertyStructure(prop);
	}
}

void Model2X::init() {
	numNonDet = 1;
	luaState = luaL_newstate();
	luaL_openlibs(luaState);
	enumerateVariables();
	maxSuccStates = 0;
	initStates = NULL;
	for (unsigned gt_nr = 0; gt_nr < model->getNumCommands(); gt_nr++) {
		const Command &gt = model->getCommand(gt_nr);
		maxSuccStates += gt.getNumAlternatives();
	}
	exprNumbers = NULL;
	intBits = sizeof(int) * 8;
	succStatesList = new unsigned[maxSuccStates];
	succRatesList = new ValueType[maxSuccStates];
	if (useRewards) {
		succRewardsList = new ValueType[maxSuccStates];
	} else {
		succRewardsList = NULL;
	}
	const unsigned maxNumNonDet = model->getNumCommands();
	nonDetBounds = new unsigned[maxNumNonDet + 1];
	unsigned numBits = 0;
	for (unsigned varNr = 0; varNr < variablesInC.size(); varNr++) {
		const unsigned varBits(numVariableBits(varNr));
		if (((numBits / intBits) < ((numBits + varBits) / intBits)) && (((numBits + varBits) % intBits) != 0)) {
			numBits = (numBits / intBits) * intBits + intBits;
		}

		numBits += varBits;
	}
	encStateSize = (numBits / intBits) + ((numBits % intBits == 0) ? 0 : 1);
	encDecSupport = new unsigned[variablesInC.size()];
	numBits = 0;
	for (unsigned varNr(0); varNr < variablesInC.size(); varNr++) {
		Expr &var = variables[varNr];
		const unsigned varBits(numVariableBits(varNr));
		if (((numBits / intBits) < ((numBits + varBits) / intBits)) && (((numBits + varBits) % intBits) != 0)) {
			unsigned numFillBits((numBits / intBits) * intBits + intBits - numBits);
			encDecSupport[varNr - 1] += numFillBits;
			numBits += numFillBits;
		}
		numBits += varBits;

		if (BoolVar == var.getVarType()) {
			encDecSupport[varNr] = numeric_limits<unsigned>::max();
		} else {
			encDecSupport[varNr] = varBits;
		}
	}

	struct hashstate hs;
	hs.size = encStateSize;
	hs.stateList = &stateList;
	struct eqstate es;
	es.size = encStateSize;
	es.stateList = &stateList;
	StateMap newStateMap(8, hs, es, allocator<pair<const unsigned *, unsigned> >());
	stateMap = newStateMap;
	valueCompute.setStateVariables(variables);
}

/**
 * Compute initial initial states.
 * Will compute the set of initial states from initial states preciate
 * and set the list of initial states and the number of initial states.
 */
void Model2X::addInitStates() {
	delete initStates;

	Expr initStatesExpr(model->getInitial());

	InitStatesComp initComp;
	initComp.setExpr(initStatesExpr);
	initComp.setVars(variables);
	initComp.setBounds(lowerBounds, upperBounds);
	std::vector<std::vector<int> > initStatesVec;
	initComp.compInits(initStatesVec);
	vector<unsigned> initStatesVecEnc;
	lua_settop(luaState, 0);
	for (unsigned iStateNr = 0; iStateNr < initStatesVec.size(); iStateNr++) {
		for (unsigned varNr = 0; varNr < variables.size(); varNr++) {
			const Expr &var = variables[varNr];
			if (BoolVar == var.getVarType()) {
				lua_pushboolean(luaState, initStatesVec[iStateNr][varNr]);
			} else {
				lua_pushnumber(luaState, initStatesVec[iStateNr][varNr]);
			}
		}
		const unsigned initState(encodeState(1));
		initStatesVecEnc.push_back(initState);
		lua_settop(luaState, 0);
	}
	initStates = new unsigned[initStatesVecEnc.size()];
	copy(initStatesVecEnc.begin(), initStatesVecEnc.end(), initStates);
	numInitStates = initStatesVec.size();
	numStates = stateList.size() / encStateSize;
}

/**
 * Compile model.
 * Call after all relevant options have been set. After this, initial
 * states can be computed, successor states be explored, etc.
 */
void Model2X::build() {
	assert(NULL != model);
	init();
	createInterface();
	if (!lua_checkstack(luaState, 1 + maxSuccStates * (max(size_t(encStateSize), variablesInC.size()) + 1 + 1))) {
		throw model2x_error("could not extend Lua stack to size needed.");
	}
}

/**
 * Destroy Model-to-Lua converter and cleanup.
 */
Model2X::~Model2X() {
	delete exprNumbers;
	delete[] succStatesList;
	delete[] succRatesList;
	delete[] succRewardsList;
	delete[] encDecSupport;
	delete[] initStates;
	delete[] nonDetBounds;
	if (NULL != luaState) {
		lua_close(luaState);
	}
}

/**
 * Add distinguished set of states.
 * To check whether a state is element of a given set of states, the
 * function @a inStateSet() may be used. State sets will be given
 * ascending numbers starting with 0.
 *
 * @param stateSet state set to be specified
 */
void Model2X::addStateSet(const Expr &stateSet) {
	stateSets.push_back(stateSet);
}

/**
 * Creates Lua interface.
 */
void Model2X::createInterface() {
	stringstream stream;

	stream << "local bit = require(\"bit\")\n";
	printGetStateSuccessors(stream);
	printInStateSet(stream);
#ifndef NDEBUG
	ofstream dbgstream("debugout.lua");
	dbgstream << stream.str() << endl;
#endif
	int error = luaL_dostring(luaState, stream.str().c_str());
	if (0 != error) {
		const string errorString(lua_tostring(luaState, -1));
		throw model2x_error("Error compiling Lua script: \"" + errorString + "\"");
	}
	currState.resize(variables.size());
	buildRateAndRewardValueTable();
}

/**
 * Print out function to check whether a state is in a given state set.
 */
void Model2X::printInStateSet(stringstream &stream) const {
	stream << "function inStateSet(setNr, ";
	printVarParams(stream);
	stream << ")\n";
	for (unsigned setNr = 0; setNr < stateSets.size(); setNr++) {
		stream << "  if ((setNr == " << setNr << ") and (" << expr2lua(stateSets[setNr].substExpr(variables, variablesInC))
				<< ")) then \n" << "    return true\n" << "  end\n";
	}
	stream << "  return false\n" << "end\n\n";
}

/**
 * Enumerate model variables.
 * Also compute position in state vector, number of bits neeed, etc. if
 * this is neccessary.
 */
void Model2X::enumerateVariables() {
	/* non-boolean */
	for (unsigned varNr = 0; varNr < model->getNumVariables(); varNr++) {
		const Expr &var = model->getVariable(varNr);
		if (!model->isParameterVariable(var)) {
			if (RangeVar == var.getVarType()) {
				const string varName(var.toString());
				variables.push_back(var);
				Expr varInC(Expr::varExpr("prismvars." + varName));
				variablesInC.push_back(varInC);
				assert(1 == var.getVarLowerBoundDen());
				assert(1 == var.getVarUpperBoundDen());
				lowerBounds.push_back(var.getVarLowerBoundNum());
				upperBounds.push_back(var.getVarUpperBoundNum());
			}
		}
	}

	/* then boolean */
	for (unsigned varNr = 0; varNr < model->getNumVariables(); varNr++) {
		const Expr &var = model->getVariable(varNr);
		if (!model->isParameterVariable(var)) {
			if (BoolVar == var.getVarType()) {
				variables.push_back(var);
				const string varName(var.toString());
				Expr varInC(Expr::varExpr("prismvars." + varName));
				variablesInC.push_back(varInC);
				lowerBounds.push_back(0);
				upperBounds.push_back(1);
			}
		}
	}
}

/**
 * Print Lua function to obtain successors of a state.
 */
void Model2X::printGetStateSuccessors(stringstream &stream) const {
	stream << "function getStateSuccessors(";
	printVarParams(stream);
	stream << ")\n";
	stream << "  local numSuccStates = 0\n"
			"  local resultTable = {}\n";
	stream << "  local nonDetBounds = {}\n";
	stream << "  table.insert(nonDetBounds,0)\n";
	unsigned rateIndex(0);
	for (unsigned cmdNr = 0; cmdNr < model->getNumCommands(); cmdNr++) {
		const Command &cmd = model->getCommand(cmdNr);
		const Expr &guard = cmd.getGuard();
		stream << "  if (" << expr2lua(guard.substExpr(variables, variablesInC)) << ") then\n";
		stream << "    local reward__ = " << cmdNr << "\n";
		for (unsigned ass_nr = 0; ass_nr < cmd.getNumAlternatives(); ass_nr++) {
			const Alternative &ass = cmd.getAlternative(ass_nr);
			stream << "    local rate__ = ";
			stream << rateIndex;
			stream << "\n";
			printSuccStateAssignments(stream, ass);
			printEncodeState(stream);
			stream << "    numSuccStates = numSuccStates + 1\n";
			rateIndex++;
		}
		if (isNonDet()) {
			stream << "    table.insert(nonDetBounds, numSuccStates)\n";
		}
		stream << "  end\n";
	}
	if (!isNonDet()) {
		stream << "    table.insert(nonDetBounds, numSuccStates)\n";
	}
	stream << "  table.insert(resultTable, #nonDetBounds)\n";
	stream << "  for bound=1,#nonDetBounds,1 do\n";
	stream << "    table.insert(resultTable, nonDetBounds[bound])\n";
	stream << "  end\n";
	stream << "  return numSuccStates, unpack(resultTable)\n";
	stream << "end\n\n";
}

/**
 * Print variable names as parameters for a Lua function.
 *
 * @param stream stream to print into
 */
void Model2X::printVarParams(stringstream &stream) const {
	stream << "prismvars";
#if 0
	for (unsigned varNr = 0; varNr < variablesInC.size(); varNr++) {
		stream << variablesInC[varNr];
		if (varNr != variablesInC.size() - 1) {
			stream << ", ";
		}
	}
#endif
}

/**
 * Print C sequence to assign values to variables of successor state.
 *
 * @param ass alternative from which successor results
 */
void Model2X::printSuccStateAssignments(stringstream &stream, const Alternative &ass) const {
	for (unsigned varNr = 0; varNr < variables.size(); varNr++) {
		const Expr &varInC = variablesInC[varNr];
		const Expr &assExpr(ass.getAssgnToVar(variables[varNr]));
		Expr assExprInC(assExpr.substExpr(variables, variablesInC));
		stream << "    " << varInC.toString() << "P = ";
		stream << expr2lua(assExprInC);
		stream << "\n";
	}
}

/**
 * Return number of bits needed to store variables.
 * The number of bits needed to store a value assignment to a certain
 * variable is the binary logarithm of this variable, rounded up.
 *
 * @param var number of variable to compute number of bits for
 * @return number of bits needed to store values of @a var
 */
unsigned Model2X::numVariableBits(unsigned var) const {
	const unsigned numPossibleValues(1 + upperBounds[var] - lowerBounds[var]);
	unsigned numBits = 1;
	unsigned roundedSize = 2;
	while (roundedSize < numPossibleValues) {
		numBits++;
		roundedSize *= 2;
	}

	return numBits;
}

/**
 * Checks whether underlying model is non-deterministic.
 *
 * @return true if time model is noin-deterministic
 */
bool Model2X::isNonDet() const {
	return (prismparser::MDP == model->getModelType());
}

/**
 * Checks whether underlying model is continuous time or not.
 *
 * @return true if time model is continuous
 */
bool Model2X::isContTime() const {
	const ModelType type(model->getModelType());
	if (CTMC == type) {
		return true;
	} else {
		return false;
	}
}

void Model2X::buildRateAndRewardValueTable() {
	for (unsigned cmdNr = 0; cmdNr < model->getNumCommands(); cmdNr++) {
		const Command &cmd = model->getCommand(cmdNr);
		Expr assRate = 0;
		for (unsigned assNr = 0; assNr < cmd.getNumAlternatives(); assNr++) {
			const Alternative &ass = cmd.getAlternative(assNr);
			rateValues.push_back(valueCompute.addEntry(ass.getWeight()));
		}
		if (useRewards) {
			const string &action = cmd.getAction();
			Expr reward = 0;
			if ("" != action) {
				reward = model->getTransReward(rewardStruct, action);
			}
			succRewardValues.push_back(valueCompute.addEntry(reward));
		}
	}
	if (useRewards) {
		stateRewardValue = valueCompute.addEntry(model->getStateReward(rewardStruct));
	}
}

ValueType Model2X::compRateValue(unsigned rate) const {
	RationalFunction res;

	valueCompute.compute(rateValues[rate], currState, res);
	return res;
}
}
