/*
 * This file is part of PARAM.
 *
 * PARAM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PARAM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PARAM. If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009-2011 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#include <stdlib.h>
#include <sys/types.h>
#include <iostream>
#include <fstream>
#include <limits>
#include <vector>
#include "../prismparser/PRISMParser.h"
#include "../prismparser/Property.h"
#include "../prismparser/Model.h"
#include "Model2XExplorer.h"
#include "../rationalFunction/RationalFunction.h"
#include "SPMC.h"
#include "SPMDP.h"
#include "ExprToNumber.h"
#include "../model2x/Model2X.h"
#include "../../../data/GraphTypes.h"
#include "../../input/ProbReader.h"
#include "../../../data/Parameters.h"

namespace parametric {
using namespace std;
using namespace prismparser;
using namespace model2x;
using namespace rational;

using namespace rational;

Logger logger = Logger::getInstance("PrismReader");

Model2XExplorer::Model2XExplorer(IOWrapper *wrapper) {
	this->wrapper = wrapper;
	model = new Model();
	logger.setLogLevel(DEFAULT_LOG_LEVEL);
}

Model2XExplorer::~Model2XExplorer() {
	delete model;
}

/**
 * Load a PASS model and appendant property file.
 * @param model_filename model file to be loaded
 * @param property_filename property file to be loaded
 */
void Model2XExplorer::loadModel(const string &model_filename, const string &property_filename) {
	prismparser::PRISMParser p;
	p.run(model_filename, *model);
	fclose(stdin);
	model->flatten();
	model->fixDeadlocks();
	p.run(property_filename, *model);
	fclose(stdin);
	for (unsigned propNr = 0; propNr < model->getNumProperties(); propNr++) {
		props.push_back(new Property(model->getProperty(propNr)));
	}
}

void Model2XExplorer::constructMC(Model2X &model2x) {
	unsigned int numEdges = 0;
	ModelType modelType = model->getModelType();

	LOG4CPLUS_INFO(logger, "State count: " << model2x.getNumStates());
	//Initialize states vector
	for (unsigned int i = wrapper->mVertexList.size(); i <= model2x.getNumStates(); i++) {
		wrapper->mVertexList.push_back(new Vertex<PTYPE>);
		wrapper->mVertexList[i]->oindex = i;
	}

	unsigned numInitStates(model2x.getNumInitStates());
	const unsigned *init(model2x.getInitStates());
	for (unsigned initNr(0); initNr < numInitStates; initNr++) {
		unsigned initState(init[initNr]);
		pmm->addInit(initState);
	}
	//Set initial state
	unsigned int initialState = init[0];
	if (numInitStates > 1) {
		LOG4CPLUS_ERROR(logger, "Multiple initial states! Will only consider first: " << initialState);
	}
	LOG4CPLUS_TRACE(logger, "Init: " << initialState);
	wrapper->mInitialNode = wrapper->mVertexList[initialState];
	wrapper->mInitialNodeID = initialState;

	vector<unsigned> pres(init, init + numInitStates);
	vector<unsigned> next;

	const RationalFunction *succRewardsList(NULL);
	const unsigned *succStatesList(model2x.getSuccStatesList());
	const RationalFunction *succRatesList(model2x.getSuccRatesList());
	const unsigned *nonDetBounds(model2x.getNonDetBounds());
	if (useReward) {
		succRewardsList = model2x.getSuccRewardsList();
	}

	param_sccmc_io::DefaultProbReader<PTYPE> probParser;
	//Initialize parameters
	string paramString;
	for (unsigned int i = 0; i < RationalFunction::getNumSymbols(); i++) {
		paramString = RationalFunction::getSymbolName(i);
		Parameters::getInstance().addVariable(paramString);
	}

	unsigned numStates(model2x.getNumStates());
	for (unsigned state(0); state < numStates; state++) {
		for (unsigned ap(0); ap < exprToNumber->getNumExprs(); ap++) {
			pmm->setAP(state, ap, model2x.inStateSet(state, ap));
			LOG4CPLUS_TRACE(logger, "State: " << state << " AP: " << ap);
		}
		model2x.getStateSuccessors(state);
		map<PMM::state, RationalFunction> probMap;
		unsigned lastNonDet(0);
		unsigned numSuccStates(model2x.getNumSuccStates());
		RationalFunction sumRate(0);
		if (CTMC == modelType) {
			for (unsigned succNr(0); succNr < numSuccStates; succNr++) {
				sumRate += succRatesList[succNr];
			}
		} else {
			sumRate = 1;
		}
		if (useTime) {
			pmm->addTime(1 / sumRate);
		}
		RationalFunction reward;
		if (useReward) {
			reward = succRewardsList[lastNonDet];
		}
		for (unsigned succNr(0); succNr < numSuccStates; succNr++) {
			unsigned succState(succStatesList[succNr]);
			RationalFunction prob(succRatesList[succNr]);
			probMap[succState] += prob;

			//Initialize edges
			string sProb = prob.toString();
			PTYPE probability = probParser.readProb(sProb);

			Edge<PTYPE> *tempedge = new Edge<PTYPE>(wrapper->mVertexList[state], wrapper->mVertexList[succState], probability);
			tempedge->not_m_min = true;
			numEdges++;
			LOG4CPLUS_TRACE(logger, "Edge: " << state << "->" << succState << ": " << probability);

			if (nonDetBounds[lastNonDet + 1] == succNr + 1) {
				lastNonDet++;
				for (map<PMM::state, RationalFunction>::iterator it(probMap.begin()); it != probMap.end(); it++) {
					unsigned succState(it->first);
					RationalFunction prob(it->second);
					pmm->addSucc(succState, prob);
				}
				if (useReward) {
					pmm->addReward(reward);
					reward = succRewardsList[lastNonDet];
				}
				probMap.clear();
				if (PMM::PMDP == pmm->getModelType()) {
					pmdp->finishChoice();
				}
			}
		}
		pmm->finishState();
	}
	LOG4CPLUS_INFO(logger, "Edge count: " << numEdges);
	wrapper->mEdgeCount = numEdges;
	LOG4CPLUS_DEBUG(logger, Parameters::getInstance().toString());
}

void Model2XExplorer::exploreAllStates(Model2X &model2x) {
	unsigned numInitStates(model2x.getNumInitStates());
	const unsigned *init(model2x.getInitStates());
	vector<unsigned> pres(init, init + numInitStates);
	vector<unsigned> next;

	unsigned numTrans(0);
	unsigned numChoices(0);
	const unsigned *succStatesList(model2x.getSuccStatesList());
	do {
		for (unsigned stateNr(0); stateNr < pres.size(); stateNr++) {
			const unsigned state(pres[stateNr]);
			const unsigned lastNumStates(model2x.getNumStates());
			model2x.getStateSuccessors(state);
			numTrans += model2x.getNumSuccStates();
			set<PMM::state> succStates(succStatesList, succStatesList + model2x.getNumSuccStates());
			for (set<unsigned>::iterator it(succStates.begin()); it != succStates.end(); it++) {
				const unsigned succState(*it);
				if (succState >= lastNumStates) {
					next.push_back(succState);
				}
			}
			if (PMM::PMDP == pmm->getModelType()) {
				numChoices += model2x.getNumNonDet();
			}
		}
		pres.swap(next);
		next.clear();
	} while (0 != pres.size());
	const unsigned numStates(model2x.getNumStates());
	pmm->reserveRowsMem(numStates);
	pmm->reserveColsMem(numTrans);
	pmm->reserveAPMem(numStates, exprToNumber->getNumExprs());
	if (PMM::PMDP == pmm->getModelType()) {
		pmdp->reserveChoicesMem(numChoices);
	}
	if (useReward) {
		if (PMM::PMDP == pmm->getModelType()) {
			pmm->reserveRewardsMem(numChoices);
		} else {
			pmm->reserveRewardsMem(numStates);
		}
	}
	if (useTime) {
		pmm->reserveTimesMem(numStates);
	}
}

void Model2XExplorer::checkUseReward(const Property &prop) {
	if (ReachRewProp == prop.getType()) {
		useReward = true;
		rewardStruct = prop.getRewStructNr();
	} else if (SteadySProp == prop.getType()) {
		useTime = true;
	} else if (SteadySRewProp == prop.getType()) {
		useReward = true;
		useTime = true;
		rewardStruct = prop.getRewStructNr();
	} else {
		for (unsigned childNr(0); childNr < prop.arity(); childNr++) {
			checkUseReward(prop[childNr]);
		}
	}
}

void Model2XExplorer::checkUseReward() {
	useReward = false;
	useTime = false;
	for (unsigned propNr(0); propNr < props.size(); propNr++) {
		const Property *prop = props[propNr];
		checkUseReward(*prop);
	}
}

void Model2XExplorer::embed() {
	for (PMM::state state(0); state < pmc->getNumStates(); state++) {
		RationalFunction sum(0);
		for (unsigned succ(0); succ < pmc->getNumSuccStates(state); succ++) {
			RationalFunction succProb = pmc->getSuccProb(state, succ);
			sum += succProb;
		}
		for (unsigned succ(0); succ < pmc->getNumSuccStates(state); succ++) {
			RationalFunction succProb = pmc->getSuccProb(state, succ);
			succProb /= sum;
			pmc->setSuccProb(state, succ, succProb);
		}
	}
}

void Model2XExplorer::explore() {
	loadModel(modelFilename, propertyFilename);

	for (unsigned varNr = 0; varNr < model->getNumVariables(); varNr++) {
		const Expr &var = model->getVariable(varNr);
		if (model->isParameterVariable(var)) {
			RationalFunction::addSymbol(var.toString());
			mpq_class leftBound(var.getVarLowerBoundNum());
			if (0 != var.getVarLowerBoundDen()) {
				leftBound /= var.getVarLowerBoundDen();
			}
			mpq_class rightBound(var.getVarUpperBoundNum());
			if (0 != var.getVarUpperBoundDen()) {
				rightBound /= var.getVarUpperBoundDen();
			}
			RationalFunction::setBounds(var.toString(), leftBound, rightBound);
		}
	}
	RationalFunction::start();
	if (model->getModelType() == MDP) {
		pmdp = new SPMDP();
		pmm = pmdp;
	} else {
		pmc = new SPMC();
		pmm = pmc;
	}
	Model2X model2x;
	model2x.setModel(*model);

	checkUseReward();
	for (unsigned propNr = 0; propNr < props.size(); propNr++) {
		exprToNumber->addProperty(*props[propNr]);
	}
	exprToNumber->build();
	for (unsigned exprNr(0); exprNr < exprToNumber->getNumExprs(); exprNr++) {
		model2x.addStateSet(exprToNumber->getExprByNumber(exprNr));
	}
	model2x.setUseRewards(useReward);
	model2x.setRewardStruct(rewardStruct);
	model2x.build();
	model2x.addInitStates();
	exploreAllStates(model2x);
	constructMC(model2x);
	if (model->getModelType() == CTMC) {
		embed();
	}
	if (NULL != pmc) {
		pmc->computeBackTransitions();
	}
}

}
