/*
 * This file is part of a parser for an extension of the PRISM language.
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The parser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the program this parser part of.
 * If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2011-2012 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#include <tr1/unordered_map>
#include <boost/shared_ptr.hpp>
#include <string>
#include <vector>
#include <limits>
#include "AST.h"
#include "Expr.h"
#include "Util.h"
#include "Types.h"

namespace prismparser {
using namespace std;

static inline long long gcd(long long a, long long b) {
	long long c = a % b;
	while (c != 0) {
		a = b;
		b = c;
		c = a % b;
	}
	return b;
}

static inline long long lcm(long long a, long long b) {
	return (a / gcd(a, b)) * b;
}

static inline void normalise(long long &num, long long &den) {
	if (0 == num) {
		den = 1;
	} else {
		long long sign = 1;
		if ((num < 0) && (den < 0)) {
			num *= -1;
			den *= -1;
		} else if ((num < 0) || (den < 0)) {
			sign = -1;
			if (den < 0) {
				den *= -1;
			} else {
				num *= -1;
			}
		}
		long long c = gcd(num, den);
		num /= c;
		den /= c;
		num *= sign;
	}
}

static inline void double2ratnoexpr(const string &double_val, long long &num, long long &den) {
	size_t dotpos = double_val.find(".");
	if (string::npos == dotpos) {
		num = atoll(double_val.c_str());
		den = 1;
	} else {
		string before_str = double_val.substr(0, dotpos);
		string after_str = double_val.substr(dotpos + 1);
		unsigned divpow = after_str.length();
		string num_str = before_str + after_str;
		num = atoll(num_str.c_str());
		den = 1;
		for (unsigned i = 0; i < divpow; i++) {
			den *= 10;
		}
		normalise(num, den);
	}
}

static inline void double2rat(const string &double_val, long long &num, long long &den) {
	size_t divpos = double_val.find("/");
	size_t epos = double_val.find("e");
	if (string::npos == epos) {
		epos = double_val.find("E");
	}

	if (string::npos != divpos) {
		string num_str = double_val.substr(0, divpos);
		string den_str = double_val.substr(divpos + 1);
		num = atoll(num_str.c_str());
		den = atoll(den_str.c_str());
	} else if (string::npos == epos) {
		double2ratnoexpr(double_val, num, den);
	} else {
		string mant_str = double_val.substr(0, epos);
		string expr_str = double_val.substr(epos + 1);
		long long expr = atoll(expr_str.c_str());
		double2ratnoexpr(mant_str, num, den);
		if (expr >= 0) {
			for (int i = 0; i < expr; i++) {
				num *= 10;
			}
		} else {
			expr = -expr;
			for (int i = 0; i < expr; i++) {
				den *= 10;
			}
		}
		normalise(num, den);
	}
}

class ExprContent {
	friend class Expr;
private:
	static ExprContent *fromAST(const boost::shared_ptr<prismparser_ast::Expr> &);
	static void incRecRefs(const ExprContent *);
	static void decRecRefs(const ExprContent *);
	static ExprContent *makeUnique(ExprContent *);
	static string op2string(ExprType);
	static void cleanupTable();
	static void exitCleanup();

	ExprContent();
	string optEmbrace() const;
	string toString() const;
	const ExprContent *substExpr(const ExprHashMap<Expr> &) const;
	const ExprContent *simplify() const;
	bool isConst() const;
	bool isInteger() const;

	struct hashstate {
		size_t operator()(const ExprContent &e) const {
			size_t hashval = 0;

			hashval = e.type + (hashval << 6) + (hashval << 16) - hashval;
			switch (e.type) {
			case VarExpr:
				for (unsigned charNr = 0; charNr < e.identifier.length(); charNr++) {
					hashval = e.identifier[charNr] + (hashval << 6) + (hashval << 16) - hashval;
				}
				break;
			case BoolExpr:
				hashval = e.bool_value + (hashval << 6) + (hashval << 16) - hashval;
				break;
			case RatExpr:
				hashval = e.num_value + (hashval << 6) + (hashval << 16) - hashval;
				hashval = e.den_value + (hashval << 6) + (hashval << 16) - hashval;
			default:
				break;
			}

			for (unsigned i = 0; i < e.children.size(); i++) {
				hashval = reinterpret_cast<size_t>(e.children[i]) + (hashval << 6) + (hashval << 16) - hashval;
			}

			return hashval;
		}
	};
	struct eqstate {
		bool operator()(const ExprContent &e1, const ExprContent &e2) const {
			if (e1.type != e2.type) {
				return false;
			}

			switch (e1.type) {
			case VarExpr:
				if (e1.identifier != e2.identifier) {
					return false;
				}
				break;
			case BoolExpr:
				if (e1.bool_value != e2.bool_value) {
					return false;
				}
				break;
			case RatExpr:
				if (e1.num_value != e2.num_value) {
					return false;
				} else if (e1.den_value != e2.den_value) {
					return false;
				}
				break;
			default:
				break;
			}

			if (e1.children.size() != e2.children.size()) {
				return false;
			}
			for (unsigned childNr = 0; childNr < e1.children.size(); childNr++) {
				if (e1.children[childNr] != e2.children[childNr]) {
					return false;
				}
			}
			return true;
		}
	};
	typedef std::tr1::unordered_map<const ExprContent, const Expr, hashstate, eqstate> UniqTab;
	static UniqTab *uniqTabPtr;
	static UniqTab &uniqTab;
	static int cleanupint;
	static ExprHashMap<pair<pair<int, int>, pair<int, int> > > bounds;
	static std::tr1::unordered_map<VarType, const string, std::tr1::hash<unsigned> > varTypeStrings;
	static map<string, VarType> varTypes;

	ExprType type;
	string identifier;
	union {
		bool bool_value;
		long long num_value;
	};
	long long den_value;
	vector<const ExprContent *> children;
	mutable unsigned refs;
};

map<string, VarType> ExprContent::varTypes;

ExprContent::UniqTab *ExprContent::uniqTabPtr = new ExprContent::UniqTab();
ExprContent::UniqTab &ExprContent::uniqTab(*uniqTabPtr);
int ExprContent::cleanupint = atexit(ExprContent::exitCleanup);
ExprHashMap<pair<pair<int, int>, pair<int, int> > > ExprContent::bounds;

static std::tr1::unordered_map<VarType, const string, std::tr1::hash<unsigned> > prepareVarTypeStrings() {
	std::tr1::unordered_map<VarType, const string, std::tr1::hash<unsigned> > result;

	result.insert(make_pair(NullVar, "null"));
	result.insert(make_pair(BoolVar, "bool"));
	result.insert(make_pair(IntVar, "int"));
	result.insert(make_pair(RangeVar, "range"));
	result.insert(make_pair(RealVar, "real"));

	return result;
}

std::tr1::unordered_map<VarType, const string, std::tr1::hash<unsigned> > ExprContent::varTypeStrings = prepareVarTypeStrings();

ExprContent::ExprContent() {
	identifier = "";
	bool_value = false;
	num_value = 0;
	den_value = 1;
	refs = 0;
}

void ExprContent::exitCleanup() {
	for (UniqTab::iterator it = uniqTab.begin(); it != uniqTab.end(); it++) {
		((Expr &) it->second).content = NULL;
	}
	delete uniqTabPtr;
}

void ExprContent::incRecRefs(const ExprContent *e) {
	assert(NULL != e);
	for (unsigned childNr = 0; childNr < e->children.size(); childNr++) {
		ExprContent::incRecRefs(e->children[childNr]);
	}
	e->refs++;
}

void ExprContent::decRecRefs(const ExprContent *e) {
	assert(NULL != e);
	for (unsigned childNr = 0; childNr < e->children.size(); childNr++) {
		ExprContent::decRecRefs(e->children[childNr]);
	}
	e->refs--;
	if (0 == e->refs) {
		UniqTab::iterator it = uniqTab.find(*e);
		((Expr &) it->second).content = NULL;
		uniqTab.erase(it);
	}
}

ExprContent *ExprContent::makeUnique(ExprContent *cont) {
	assert(NULL != cont);
	ExprContent *res;

	UniqTab::iterator it = uniqTab.find(*cont);
	if (it != uniqTab.end()) {
		res = (ExprContent *) &(it->first);
	} else {
		Expr defaultExpr;

		uniqTab.insert(make_pair(*cont, defaultExpr));
		UniqTab::iterator it = uniqTab.find(*cont);
		res = (ExprContent *) &(it->first);
		((Expr &) it->second).content = res;
	}
	if (res != cont) {
		delete cont;
	}

	return res;
}

ExprContent *ExprContent::fromAST(const boost::shared_ptr<prismparser_ast::Expr> &ast) {
	ExprContent *cont = new ExprContent();

	for (unsigned childNr = 0; childNr < ast->arity(); childNr++) {
		cont->children.push_back(fromAST(ast->children[childNr]));
	}

	switch (ast->getKind()) {
	case prismparser_ast::Null:
		assert(false);
		break;

	case prismparser_ast::Var:
		cont->type = VarExpr;
		break;
	case prismparser_ast::Bool:
		cont->type = BoolExpr;
		break;
	case prismparser_ast::Int:
		cont->type = RatExpr;
		break;
	case prismparser_ast::Rat:
		cont->type = RatExpr;
		break;
	case prismparser_ast::Double:
		cont->type = RatExpr;
		break;

	case prismparser_ast::Not:
		cont->type = NotExpr;
		break;
	case prismparser_ast::And:
		cont->type = AndExpr;
		break;
	case prismparser_ast::Or:
		cont->type = OrExpr;
		break;
	case prismparser_ast::Impl:
		cont->type = ImplExpr;
		break;

	case prismparser_ast::Eq:
		cont->type = EqExpr;
		break;
	case prismparser_ast::Neq:
		cont->type = NeqExpr;
		break;
	case prismparser_ast::Lt:
		cont->type = LtExpr;
		break;
	case prismparser_ast::Gt:
		cont->type = GtExpr;
		break;
	case prismparser_ast::Le:
		cont->type = LeExpr;
		break;
	case prismparser_ast::Ge:
		cont->type = GeExpr;
		break;

	case prismparser_ast::Plus:
		cont->type = PlusExpr;
		break;
	case prismparser_ast::Minus:
		cont->type = MinusExpr;
		break;
	case prismparser_ast::Uminus:
		cont->type = UminusExpr;
		break;
	case prismparser_ast::Mult:
		cont->type = MultExpr;
		break;
	case prismparser_ast::Div:
		cont->type = DivExpr;
		break;
	case prismparser_ast::Pow:
		cont->type = PowExpr;
		break;

	case prismparser_ast::Ite:
		cont->type = IteExpr;
		break;
	case prismparser_ast::Min:
		cont->type = MinExpr;
		break;
	case prismparser_ast::Max:
		cont->type = MaxExpr;
		break;
	case prismparser_ast::Floor:
		cont->type = FloorExpr;
		break;
	case prismparser_ast::Ceil:
		cont->type = CeilExpr;
		break;
	case prismparser_ast::MinusInf:
		cont->type = RatExpr;
		cont->num_value = -1;
		cont->den_value = 0;
		break;
	case prismparser_ast::PlusInf:
		cont->type = RatExpr;
		cont->num_value = +1;
		cont->den_value = 0;
		break;
	default:
		assert(false);
	}

	if (prismparser_ast::Bool == ast->getKind()) {
		cont->bool_value = ast->getBool();
	} else if (prismparser_ast::Int == ast->getKind()) {
		cont->num_value = ast->getInt();
		cont->type = RatExpr;
	} else if (prismparser_ast::Double == ast->getKind()) {
		double2rat(ast->getDoubleAsString(), cont->num_value, cont->den_value);
	} else if (prismparser_ast::Var == ast->getKind()) {
		cont->identifier = ast->getIdentifier();
	}
	cont = makeUnique(cont);

	return cont;
}

bool ExprContent::isConst() const {
	return ((BoolExpr == type) || (RatExpr == type));
}

bool ExprContent::isInteger() const {
	return ((RatExpr == type) && (1 == den_value));
}

string ExprContent::op2string(ExprType type) {
	switch (type) {
	case NotExpr:
		return "!";
	case AndExpr:
		return "&";
	case OrExpr:
		return "|";
	case ImplExpr:
		return "=>";
	case EqExpr:
		return "=";
	case NeqExpr:
		return "!=";
	case LtExpr:
		return "<";
	case GtExpr:
		return ">";
	case LeExpr:
		return "<=";
	case GeExpr:
		return ">=";
	case PlusExpr:
		return "+";
	case MinusExpr:
		return "-";
	case UminusExpr:
		return "-";
	case MultExpr:
		return "*";
	case DivExpr:
		return "/";
	default:
		assert(false);
	}
}

string ExprContent::optEmbrace() const {
	string result;
	if ((VarExpr == type) || (NotExpr == type) || (PowExpr == type) || (MinExpr == type) || (MaxExpr == type)) {
		return toString();
	} else {
		if (children.size() > 1) {
			result += "(";
		}
		result += toString();
		if (children.size() > 1) {
			result += ")";
		}
	}
	return result;
}

string ExprContent::toString() const {
	string result;
	switch (type) {
	case VarExpr:
		result = identifier;
		break;
	case BoolExpr:
		result = bool_value ? "true" : "false";
		break;
	case RatExpr:
		if (1 == den_value) {
			result = intToString(num_value);
		} else {
			result = intToString(num_value) + "/" + intToString(den_value);
		}
		break;
	case NotExpr:
		result += "!" + children[0]->optEmbrace();
		break;
	case PowExpr:
		result = "pow(" + children[0]->toString() + "," + children[1]->toString() + ")";
	case IteExpr:
		result = children[0]->optEmbrace() + "?" + children[1]->optEmbrace() + ":" + children[2]->optEmbrace();
		break;
	case MinExpr:
		result = "min(" + children[0]->toString() + "," + children[1]->toString() + ")";
		break;
	case MaxExpr:
		result = "max(" + children[0]->toString() + "," + children[1]->toString() + ")";
		break;
	case FloorExpr:
		result = "floor(" + children[0]->toString() + ")";
		break;
	case CeilExpr:
		result = "ceil(" + children[0]->toString() + ")";
		break;
	case UminusExpr:
		result = "-" + children[0]->optEmbrace();
		break;
	case AndExpr:
	case OrExpr:
	case ImplExpr:
	case EqExpr:
	case NeqExpr:
	case LtExpr:
	case GtExpr:
	case LeExpr:
	case GeExpr:
	case PlusExpr:
	case MinusExpr:
	case MultExpr:
	case DivExpr:
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			result += children[childNr]->optEmbrace();
			if (childNr < children.size() - 1) {
				result += op2string(type);
			}
		}
		break;
	}

	return result;
}

const ExprContent *ExprContent::substExpr(const ExprHashMap<Expr> &table) const {
	Expr finder;
	finder.content = this;
	ExprHashMap<Expr>::const_iterator it = table.find(finder);
	if (it != table.end()) {
		finder.content = NULL;
		return it->second.content;
	} else {
		ExprContent *res = new ExprContent;
		res->identifier = identifier;
		res->bool_value = bool_value;
		res->num_value = num_value;
		res->den_value = den_value;
		res->type = type;
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			res->children.push_back(children[childNr]->substExpr(table));
		}
		res = makeUnique(res);
		finder.content = NULL;
		return res;
	}
}

void ExprContent::cleanupTable() {
	vector<UniqTab::iterator> remove;
	for (UniqTab::iterator it = uniqTab.begin(); it != uniqTab.end(); it++) {
		if (0 == it->first.refs) {
			remove.push_back(it);
		}
	}
	for (unsigned remNr = 0; remNr < remove.size(); remNr++) {
		uniqTab.erase(remove[remNr]);
	}
}

const ExprContent *ExprContent::simplify() const {
	ExprContent *res = new ExprContent;
	res->identifier = identifier;
	res->bool_value = bool_value;
	res->num_value = num_value;
	res->den_value = den_value;
	res->type = type;

	switch (type) {
	case VarExpr:
	case BoolExpr:
	case RatExpr:
		delete res;
		return this;
	case NotExpr: {
		const ExprContent *c = children[0]->simplify();
		if (!c->isConst()) {
			res->children.push_back(c);
		} else {
			res->type = BoolExpr;
			res->bool_value = !c->bool_value;
		}
		break;
	}
	case AndExpr: {
		vector<const ExprContent *> cs;
		bool oneFalse = false;
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			const ExprContent *c = children[childNr]->simplify();
			if (!c->isConst()) {
				cs.push_back(c);
			} else if (!c->bool_value) {
				oneFalse = true;
			}
		}
		if (oneFalse) {
			res->type = BoolExpr;
			res->bool_value = false;
		} else if (0 == cs.size()) {
			res->type = BoolExpr;
			res->bool_value = true;
		} else if (1 == cs.size()) {
			delete res;
			return (ExprContent *) cs[0];
		} else {
			for (unsigned childNr = 0; childNr < cs.size(); childNr++) {
				res->children.push_back(cs[childNr]);
			}
		}
		break;
	}
	case OrExpr: {
		vector<const ExprContent *> cs;
		bool oneTrue = false;
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			const ExprContent *c = children[childNr]->simplify();
			if (!c->isConst()) {
				cs.push_back(c);
			} else if (c->bool_value) {
				oneTrue = true;
			}
		}
		if (oneTrue) {
			res->type = BoolExpr;
			res->bool_value = true;
		} else if (0 == cs.size()) {
			res->type = BoolExpr;
			res->bool_value = false;
		} else if (1 == cs.size()) {
			delete res;
			return (ExprContent *) cs[0];
		} else {
			for (unsigned childNr = 0; childNr < cs.size(); childNr++) {
				res->children.push_back(cs[childNr]);
			}
		}
		break;
	}
	case ImplExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && c2->isConst()) {
			res->type = BoolExpr;
			res->bool_value = !c1->bool_value || c2->bool_value;
		} else if (c1->isConst()) {
			if (c1->bool_value) {
				delete res;
				res = (ExprContent *) c2;
			} else {
				res->type = BoolExpr;
				res->bool_value = true;
			}
		} else if (c2->isConst()) {
			if (c2->bool_value) {
				res->type = BoolExpr;
				res->bool_value = true;
			} else {
				res->type = NotExpr;
				res->children.push_back(c1);
			}
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case EqExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 != c2) {
			if (c1->isConst() && c2->isConst()) {
				res->type = BoolExpr;
				res->bool_value = false;
			} else {
				res->children.push_back(c1);
				res->children.push_back(c2);
			}
		} else {
			res->type = BoolExpr;
			res->bool_value = true;
		}
		break;
	}
	case NeqExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 == c2) {
			res->type = BoolExpr;
			res->bool_value = false;
		} else {
			if (c1->isConst() && c2->isConst()) {
				res->type = BoolExpr;
				res->bool_value = true;
			} else {
				res->children.push_back(c1);
				res->children.push_back(c2);
			}
		}
		break;
	}
	case LtExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 == c2) {
			res->type = BoolExpr;
			res->bool_value = false;
		} else if (c1->isConst() && c2->isConst()) {
			res->type = BoolExpr;
			res->bool_value = (c1->num_value * c2->den_value < c2->num_value * c1->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case GtExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 == c2) {
			res->type = BoolExpr;
			res->bool_value = false;
		} else if (c1->isConst() && c2->isConst()) {
			res->type = BoolExpr;
			res->bool_value = (c1->num_value * c2->den_value > c2->num_value * c1->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case LeExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 == c2) {
			res->type = BoolExpr;
			res->bool_value = true;
		} else if (c1->isConst() && c2->isConst()) {
			res->type = BoolExpr;
			res->bool_value = (c1->num_value * c2->den_value <= c2->num_value * c1->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case GeExpr: {
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1 == c2) {
			res->type = BoolExpr;
			res->bool_value = true;
		} else if (c1->isConst() && c2->isConst()) {
			res->type = BoolExpr;
			res->bool_value = (c1->num_value * c2->den_value >= c2->num_value * c1->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case PlusExpr: {
		vector<const ExprContent *> cs;
		long long values = 0;
		long long lmult = 1;
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			const ExprContent *c = children[childNr]->simplify();
			if (RatExpr != c->type) {
				cs.push_back(c);
			} else {
				lmult = lcm(lmult, c->den_value);
			}
		}
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			const ExprContent *c = children[childNr]->simplify();
			if (RatExpr == c->type) {
				values += (lmult / c->den_value) * c->num_value;
			}
		}
		if (0 == cs.size()) {
			res->type = RatExpr;
			res->num_value = values;
			res->den_value = lmult;
		} else if (1 == cs.size()) {
			if (0 == values) {
				delete res;
				res = (ExprContent *) cs[0];
			} else {
				res->children.push_back(cs[0]);
				ExprContent *rest = new ExprContent;
				rest->type = RatExpr;
				rest->num_value = values;
				rest->den_value = lmult;
				rest = makeUnique(rest);
				res->children.push_back(rest);
			}
		} else {
			for (unsigned childNr = 0; childNr < cs.size(); childNr++) {
				res->children.push_back(cs[childNr]);
			}
			if (0 != values) {
				ExprContent *rest = new ExprContent;
				rest->type = RatExpr;
				rest->num_value = values;
				rest->den_value = lmult;
				rest = makeUnique(rest);
				res->children.push_back(rest);
			}
		}
		break;
	}
	case UminusExpr: {
		assert(1 == children.size());
		const ExprContent *c = children[0]->simplify();
		if (c->isConst()) {
			res->type = RatExpr;
			res->num_value = -c->num_value;
			res->den_value = c->den_value;
		} else {
			res->children.push_back(children[0]->simplify());
		}
		break;
	}
	case MinusExpr: {
		assert(2 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && c2->isConst()) {
			res->type = RatExpr;
			res->num_value = c1->num_value * c2->den_value - c2->num_value * c1->den_value;
			res->den_value = c1->den_value * c2->den_value;
			normalise(res->num_value, res->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case MultExpr: {
		vector<const ExprContent *> cs;
		long long num = 1;
		long long den = 1;
		bool foundZero = false;
		for (unsigned childNr = 0; childNr < children.size(); childNr++) {
			const ExprContent *c = children[childNr]->simplify();
			if (RatExpr != c->type) {
				cs.push_back(c);
			} else {
				num *= c->num_value;
				den *= c->den_value;
				normalise(num, den);
				if (0 == num) {
					foundZero = true;
				}
			}
		}
		if (foundZero) {
			res->type = RatExpr;
			res->num_value = 0;
			res->den_value = 1;
		} else if (0 == cs.size()) {
			res->type = RatExpr;
			res->num_value = num;
			res->den_value = den;
		} else if (1 == cs.size()) {
			if ((1 == num) && (1 == den)) {
				delete res;
				res = (ExprContent *) cs[0];
			} else {
				res->children.push_back(cs[0]);
				ExprContent *rest = new ExprContent;
				rest->type = RatExpr;
				rest->num_value = num;
				rest->den_value = den;
				rest = makeUnique(rest);
				res->children.push_back(rest);
			}
		} else {
			for (unsigned childNr = 0; childNr < cs.size(); childNr++) {
				res->children.push_back(cs[childNr]);
			}
			if ((1 != num) || (1 != den)) {
				ExprContent *rest = new ExprContent;
				rest->type = RatExpr;
				rest->num_value = num;
				rest->den_value = den;
				rest = makeUnique(rest);
				res->children.push_back(rest);
			}
		}
		break;
	}
	case DivExpr: {
		assert(2 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && (0 == c1->num_value)) {
			delete res;
			res = (ExprContent *) c1;
		} else if (c1->isConst() && c2->isConst()) {
			res->type = RatExpr;
			res->num_value = c1->num_value * c2->den_value;
			res->den_value = c1->den_value * c2->num_value;
			normalise(res->num_value, res->den_value);
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case PowExpr: {
		assert(2 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && c2->isInteger()) {
			long long times = c2->num_value;
			int sign = times < 0 ? -1 : 1;
			times *= sign;
			long long num = 1;
			long long den = 1;
			for (long long i = 0; i < times; i++) {
				num *= c1->num_value;
				den *= c1->den_value;
			}
			res->type = RatExpr;
			if (-1 == sign) {
				res->num_value = den;
				res->den_value = num;
			} else {
				res->num_value = num;
				res->den_value = den;
			}
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case IteExpr: {
		assert(3 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		const ExprContent *c3 = children[2]->simplify();
		if (c1->isConst()) {
			delete res;
			if (c1->bool_value) {
				res = (ExprContent *) c2;
			} else {
				res = (ExprContent *) c3;
			}
		} else if (c2 == c3) {
			delete res;
			res = (ExprContent *) c2;
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
			res->children.push_back(c3);
		}
		break;
	}
	case MinExpr: {
		assert(2 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && c2->isConst()) {
			res->type = RatExpr;
			delete res;
			if (c1->num_value * c2->den_value < c2->num_value * c2->den_value) {
				res = (ExprContent *) c1;
			} else {
				res = (ExprContent *) c2;
			}
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case MaxExpr: {
		assert(2 == children.size());
		const ExprContent *c1 = children[0]->simplify();
		const ExprContent *c2 = children[1]->simplify();
		if (c1->isConst() && c2->isConst()) {
			res->type = RatExpr;
			delete res;
			if (c1->num_value * c2->den_value > c2->num_value * c2->den_value) {
				res = (ExprContent *) c1;
			} else {
				res = (ExprContent *) c2;
			}
		} else {
			res->children.push_back(c1);
			res->children.push_back(c2);
		}
		break;
	}
	case FloorExpr: {
		assert(1 == children.size());
		const ExprContent *c = children[0]->simplify();
		if (c->isConst()) {
			res->type = RatExpr;
			res->den_value = 1;
			res->num_value = c->num_value / c->den_value;
			if ((c->num_value < 0) && (c->num_value % c->den_value != 0)) {
				res->num_value--;
			}
		} else {
			res->children.push_back(children[0]->simplify());
		}
		break;
	}
	case CeilExpr: {
		assert(1 == children.size());
		const ExprContent *c = children[0]->simplify();
		if (c->isConst()) {
			res->type = RatExpr;
			res->den_value = 1;
			res->num_value = c->num_value / c->den_value;
			if ((c->num_value > 0) && (c->num_value % c->den_value != 0)) {
				res->num_value++;
			}
		} else {
			res->children.push_back(children[0]->simplify());
		}
		break;
	}
	}

	res = makeUnique(res);
	return res;
}

Expr::Expr() {
	content = NULL;
}

Expr::Expr(const boost::shared_ptr<prismparser_ast::Expr> &ast) {
	content = ExprContent::fromAST(ast);
	ExprContent::incRecRefs(content);
}

Expr::Expr(const Expr &_expr) {
	content = _expr.content;
	if (NULL != content) {
		ExprContent::incRecRefs(content);
	}
}

Expr::Expr(int value) {
	ExprContent *cont = new ExprContent();
	cont->type = RatExpr;
	cont->num_value = value;
	cont->den_value = 1;
	content = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(content);
}

Expr::~Expr() {
	if (NULL != content) {
		ExprContent::decRecRefs(content);
	}
}

unsigned Expr::arity() const {
	assert(NULL != content);
	return content->children.size();
}

const Expr &Expr::operator[](const unsigned index) const {
	assert(index < content->children.size());
	assert(ExprContent::uniqTab.end() != ExprContent::uniqTab.find(*content->children[index]));
	return ExprContent::uniqTab.find(*content->children[index])->second;
}

Expr Expr::plusExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = PlusExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::multExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = MultExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::divExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = DivExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::andExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = AndExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::andExpr(const vector<Expr> &es) {
	ExprContent *cont = new ExprContent();
	if (0 == es.size()) {
		cont->type = BoolExpr;
		cont->bool_value = true;
	} else if (1 == es.size()) {
		delete cont;
		assert(NULL != es[0].content);
		cont = (ExprContent *) es[0].content;
	} else {
		cont->type = AndExpr;
		for (unsigned eNr = 0; eNr < es.size(); eNr++) {
			assert(NULL != es[eNr].content);
			cont->children.push_back(es[eNr].content);
		}
	}
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::orExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = OrExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::orExpr(const vector<Expr> &es) {
	ExprContent *cont = new ExprContent();
	if (0 == es.size()) {
		cont->type = BoolExpr;
		cont->bool_value = false;
	} else if (1 == es.size()) {
		delete cont;
		assert(NULL != es[0].content);
		cont = (ExprContent *) es[0].content;
	} else {
		cont->type = OrExpr;
		for (unsigned eNr = 0; eNr < es.size(); eNr++) {
			assert(NULL != es[eNr].content);
			cont->children.push_back(es[eNr].content);
		}
	}
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::implExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = ImplExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::notExpr(const Expr &e) {
	assert(e.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = NotExpr;
	cont->children.push_back(e.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::iteExpr(const Expr &eIf, const Expr &eThen, const Expr &eElse) {
	assert(eIf.content != NULL);
	assert(eThen.content != NULL);
	assert(eElse.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = IteExpr;
	cont->children.push_back(eIf.content);
	cont->children.push_back(eThen.content);
	cont->children.push_back(eElse.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::sumExpr(const vector<Expr> &summands) {
	ExprContent *cont = new ExprContent();
	if (0 == summands.size()) {
		cont->type = RatExpr;
		cont->num_value = 0;
		cont->den_value = 1;
	} else {
		cont->type = PlusExpr;
		for (unsigned sumNr = 0; sumNr < summands.size(); sumNr++) {
			cont->children.push_back(summands[sumNr].content);
		}
	}
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::varExpr(const string &__identifier) {
	assert("" != __identifier);
	ExprContent *cont = new ExprContent();
	cont->type = VarExpr;
	cont->identifier = __identifier;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::eqExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = EqExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::ltExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = LtExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::gtExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = GtExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::leExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = LeExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::geExpr(const Expr &e1, const Expr &e2) {
	assert(e1.content != NULL);
	assert(e2.content != NULL);
	ExprContent *cont = new ExprContent();
	cont->type = GeExpr;
	cont->children.push_back(e1.content);
	cont->children.push_back(e2.content);
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::falseExpr() {
	ExprContent *cont = new ExprContent();
	cont->type = BoolExpr;
	cont->bool_value = false;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::trueExpr() {
	ExprContent *cont = new ExprContent();
	cont->type = BoolExpr;
	cont->bool_value = true;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::ratExpr(long long num, long long den) {
	assert(0 != den);
	ExprContent *cont = new ExprContent();

	normalise(num, den);
	cont->type = RatExpr;
	cont->num_value = num;
	cont->den_value = den;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::ratExprInt(int num, int den) {
	assert(0 != den);
	ExprContent *cont = new ExprContent();

	long long longnum = num;
	long long longden = den;
	normalise(longnum, longden);
	num = longnum;
	den = longden;
	cont->type = RatExpr;
	cont->num_value = num;
	cont->den_value = den;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::ratExpr(const string &str, int base) {
	assert(10 == base);
	ExprContent *cont = new ExprContent();
	cont->type = RatExpr;
	cont->num_value = 0;
	long long digVal = 1;
	for (int charPos = str.size(); charPos > 0; charPos--) {
		digVal *= base;
		cont->num_value += (str[charPos] - '0');
	}
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::ratExpr(const string &str) {
	assert("" != str);
	ExprContent *cont = new ExprContent();
	double2rat(str, cont->num_value, cont->den_value);
	cont->type = RatExpr;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::minusInfExpr() {
	ExprContent *cont = new ExprContent();
	cont->num_value = -1;
	cont->den_value = 0;
	cont->type = RatExpr;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

Expr Expr::plusInfExpr() {
	ExprContent *cont = new ExprContent();
	cont->num_value = +1;
	cont->den_value = 0;
	cont->type = RatExpr;
	cont = ExprContent::makeUnique(cont);
	ExprContent::incRecRefs(cont);
	return ExprContent::uniqTab[*cont];
}

bool Expr::isMinusInf() const {
	return ((RatExpr == content->type) && (-1 == content->num_value) && (0 == content->den_value));
}

bool Expr::isPlusInf() const {
	return ((RatExpr == content->type) && (+1 == content->num_value) && (0 == content->den_value));
}

string Expr::toString() const {
	if (NULL == content) {
		return "Null";
	} else {
		return content->toString();
	}
}

bool Expr::isNull() const {
	return (NULL == content);
}

const string &Expr::getName() const {
	return content->identifier;
}

bool Expr::isBoolConst() const {
	return (BoolExpr == content->type);
}

bool Expr::isITE() const {
	return (IteExpr == content->type);
}

bool Expr::isRational() const {
	if (NULL == content) {
		return false;
	}
	return (RatExpr == content->type);
}

bool Expr::isInteger() const {
	return (isRational() && (1 == content->den_value));
}

bool Expr::isFalse() const {
	return ((BoolExpr == content->type) && !content->bool_value);
}

bool Expr::isTrue() const {
	return ((BoolExpr == content->type) && content->bool_value);
}

bool Expr::isMult() const {
	return MultExpr == content->type;
}

Expr Expr::simplify() const {
	const ExprContent *resCont = this->content->simplify();
	ExprContent::incRecRefs(resCont);
	ExprContent::cleanupTable();
	return ExprContent::uniqTab[*resCont];
}

Expr Expr::substExpr(const ExprHashMap<Expr> &table) const {
	const ExprContent *resCont = content->substExpr(table);
	ExprContent::incRecRefs(resCont);
	return ExprContent::uniqTab[*resCont];
}

Expr Expr::substExpr(const vector<Expr> &from, const vector<Expr> &to) const {
	assert(from.size() == to.size());
	ExprHashMap<Expr> table;
	for (unsigned exprNr = 0; exprNr < from.size(); exprNr++) {
		table.insert(make_pair(from[exprNr], to[exprNr]));
	}
	const ExprContent *resCont = content->substExpr(table);
	ExprContent::incRecRefs(resCont);
	return ExprContent::uniqTab[*resCont];
}

long long Expr::getNumerator() const {
	return content->num_value;
}

long long Expr::getDenominator() const {
	return content->den_value;
}

double Expr::getRatAsDouble() const {
	if (0 == content->den_value) {
		if (-1 == content->num_value) {
			return -numeric_limits<double>::infinity();
		} else if (+1 == content->num_value) {
			return +numeric_limits<double>::infinity();
		} else {
			assert(false);
		}
	} else {
		return ((double) content->num_value) / ((double) content->den_value);
	}
}

bool Expr::isVar() const {
	return (VarExpr == content->type);
}

ExprType Expr::getType() const {
	return content->type;
}

ostream &operator<<(ostream &stream, const Expr &e) {
	stream << e.toString();
	return stream;
}

bool operator==(const Expr &e1, const Expr &e2) {
	return (e1.content == e2.content);
}

Expr &Expr::operator=(const Expr &from) {
	if (NULL != from.content) {
		ExprContent::incRecRefs(from.content);
		if (NULL != content) {
			ExprContent::decRecRefs(content);
		}
		content = from.content;
	}

	return *this;
}

void Expr::setVarType(VarType varType) const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	ExprContent::varTypes[content->identifier] = varType;
}

VarType Expr::getVarType() const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	map<string, VarType>::iterator it = ExprContent::varTypes.find(content->identifier);
	if (ExprContent::varTypes.end() == it) {
		return NullVar;
	} else {
		return it->second;
	}
}

void Expr::setVarBounds(const Expr &expr, int lowerNum, int lowerDen, int upperNum, int upperDen) {
	assert(NULL != expr.content);
	assert(VarExpr == expr.content->type);
	pair<pair<int, int>, pair<int, int> > b = make_pair(make_pair(lowerNum, lowerDen), make_pair(upperNum, upperDen));
	ExprContent::bounds.insert(make_pair(expr, b));
}

int Expr::getVarLowerBoundNum() const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	return ExprContent::bounds.find(*this)->second.first.first;
}

int Expr::getVarLowerBoundDen() const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	return ExprContent::bounds.find(*this)->second.first.second;
}

int Expr::getVarUpperBoundNum() const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	return ExprContent::bounds.find(*this)->second.second.first;
}

int Expr::getVarUpperBoundDen() const {
	assert(NULL != content);
	assert(VarExpr == content->type);
	return ExprContent::bounds.find(*this)->second.second.second;
}

const string &Expr::varTypeToString(VarType varType) {
	return ExprContent::varTypeStrings.find(varType)->second;
}
}
