/*
 * This file is part of a parser for an extension of the PRISM language.
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The parser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the program this parser part of.
 * If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2007-2010 Bjoern Wachter (Bjoern.Wachter@comlab.ox.ac.uk)
 * Copyright 2009-2012 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#include <sstream>
#include <boost/foreach.hpp>
#include "Util.h"
#include "Property.h"
#include "Model.h"
#include "ModelImpl.h"
#include "Util.h"
#include "Expr.h"
#include "System.h"
#include <iostream>

#define WITH_PARAM

namespace prismparser {
using namespace std;

ModelImpl::ModelImpl() {
	model_type = MDP;
	properties = new Properties();
	system = NULL;
	refs = 0;
}

ModelImpl::~ModelImpl() {
	if (NULL != system) {
		delete system;
	}
	delete properties;
}

unsigned ModelImpl::getNumProperties() const {
	return properties->size();
}

const string &ModelImpl::toString() const {
	if ("" == str) {
		str += "module M\n";
		for (vector<string>::const_iterator i = var_names.begin(); i != var_names.end(); i++) {
			const string & v = *i;
			tr1::unordered_map<string, Expr>::const_iterator vit = variables.find(*i);
			assert(vit != variables.end());
			str += v + " : " + Expr::varTypeToString(vit->second.getVarType()) + ";\n";
		}
		long long i = 0;
		BOOST_FOREACH (boost::shared_ptr<Command> c, guarded_transitions) {
			assert(c);
			str += "Nr. " + intToString(i) + " : " + c->toString();
			i++;
		}
		str += "endmodule\n";
		str += "init\n" + init.toString() + "\nendinit" + "\n";
		if (NULL != system) {
			stringstream stream;
			stream << *system;
			str += stream.str();
		}
		for (unsigned rewardNr = 0; rewardNr < rewards.size(); rewardNr++) {
			const Reward &reward(rewards.find(rewardNr)->second);
			str += reward.toString();
		}
		BOOST_FOREACH (boost::shared_ptr<Property> p, *properties) {
			assert(p);
			str += p->toString() + "\n";
		}
	}

	return str;
}

const Module &ModelImpl::getModule(const string & module_name) const {
	// go through the modules to find suitable one
	for (Modules::const_iterator i = modules.begin(); i != modules.end(); ++i) {
		if ((*i)->getName() == module_name) {
			return *(*i).get();
		}
	}
	assert(false);
}

void ModelImpl::addVariable(const Expr &var) {
	const string &id(var.toString());

	if (variables.find(id) != variables.end()) {
		return;
	}
	var_names.push_back(id);
	variables.insert(pair<string, Expr>(id, var));
}

void ModelImpl::computeDefaultInitialValue() {
	init = Expr::trueExpr();
	vector<Expr> init_vec;
	for (ExprHashMap<Expr>::iterator it = default_initial_value.begin(); it != default_initial_value.end(); ++it) {
		if (!isParameterVariable(it->first)) {
			if ((it->second).isBoolConst()) {
				if ((it->second).isTrue()) {
					init_vec.push_back(it->first);
				} else {
					init_vec.push_back(Expr::notExpr(it->first));
				}
			} else {
				init_vec.push_back(Expr::eqExpr(it->first, it->second));
			}
		}
	}
	init = Expr::andExpr(init_vec);
}

void ModelImpl::addAction(const Action & a) {
	assert(a != "");
	actions.insert(a);
}

void ModelImpl::addModule(Module *module) {
	assert(module);
	boost::shared_ptr<Module> mod_ptr(module);
	modules.push_back(mod_ptr);
}

void ModelImpl::setInitial(const Expr & __init) {
	init = __init;
}

void ModelImpl::addPredicate(const Expr & __expr) {
	user_predicates.push_back(__expr);
}

void ModelImpl::addProperty(Property *p) {
	assert(p);
	boost::shared_ptr<Property> ptr(p);
	properties->push_back(ptr);
}

void ModelImpl::addInvariant(const Expr & __expr) {
	invariants.push_back(__expr.simplify());
}

void ModelImpl::createDefaultSystem() {
	Modules::const_iterator mit = modules.begin();
	system = new System();
	system->type = System::IdentifierSystem;
	system->identifier = (*mit)->impl->name;

	for (++mit; mit != modules.end(); ++mit) {
		System *oldSystem = system;
		system = new System();
		system->type = System::AlparSystem;
		system->left = oldSystem;
		System *newSystem = new System();
		newSystem->type = System::IdentifierSystem;
		newSystem->identifier = (*mit)->impl->name;
		system->right = newSystem;
	}
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenIdentifier(const System &system, set<string> &mentioned) {
	ModuleImpl *impl = NULL;
	for (Modules::const_iterator it = modules.begin(); it != modules.end(); it++) {
		if ((*it)->impl->name == system.identifier) {
			impl = getModule(system.identifier).impl;
		}
	}

	if (NULL == impl) {
		throw prismparser_error("module \"" + system.identifier + "\" in system "
				"definition does not exist");
	} else if (mentioned.end() != mentioned.find(system.identifier)) {
		throw prismparser_error("module \"" + system.identifier + "\" used more "
				"than once in system definition");
	}
	mentioned.insert(system.identifier);
	return new SynchronizedCommands(impl->sync_guarded_transitions);
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenRename(const System &system, set<string> &mentioned) {
	SynchronizedCommands *inner = flatten(*system.left, mentioned);
	for (System::RenameMap::const_iterator it = system.rename.begin(); it != system.rename.end(); it++) {
		SynchronizedCommands::iterator sit = inner->find(it->first);
		if (inner->end() != sit) {
			Commands cmds = sit->second;
			inner->erase(sit);
			inner->insert(make_pair(it->second, cmds));
		}
	}
	return inner;
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenHide(const System &system, set<string> &mentioned) {
	SynchronizedCommands *inner = flatten(*system.left, mentioned);
	for (System::HideSet::const_iterator it = system.hide.begin(); it != system.hide.end(); it++) {
		SynchronizedCommands::iterator sit = inner->find(*it);
		if (inner->end() != sit) {
			Commands cmds = sit->second;
			inner->erase(sit);
			(*inner)[""].insert((*inner)[""].end(), cmds.begin(), cmds.end());
		}
	}
	return inner;
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenAlpar(const System &system, set<string> &mentioned) {
	SynchronizedCommands *left = flatten(*system.left, mentioned);
	SynchronizedCommands *right = flatten(*system.right, mentioned);
	for (SynchronizedCommands::iterator it = right->begin(); it != right->end(); it++) {
		const Action &action = it->first;
		const Commands &cmds = it->second;
		if ("" == action) {
			(*left)[""].insert((*left)[""].end(), cmds.begin(), cmds.end());
		} else {
			if (left->end() == left->find(action)) {
				left->insert(make_pair(action, cmds));
			} else {
				const Commands cmdsLeft = left->find(action)->second;
				left->erase(action);
				for (Commands::const_iterator leftIt = cmdsLeft.begin(); leftIt != cmdsLeft.end(); leftIt++) {
					for (Commands::const_iterator rightIt = cmds.begin(); rightIt != cmds.end(); rightIt++) {
						boost::shared_ptr<Command> newCmd((*(**leftIt).impl) * (*(**rightIt).impl));
						(*left)[action].push_back(newCmd);
					}
				}
			}
		}
	}
	delete right;
	return left;
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenAspar(const System &system, set<string> &mentioned) {
	SynchronizedCommands *left = flatten(*system.left, mentioned);
	SynchronizedCommands *right = flatten(*system.right, mentioned);
	for (SynchronizedCommands::iterator it = right->begin(); it != right->end(); it++) {
		const Action &action = it->first;
		(*left)[action].insert((*left)[action].end(), it->second.begin(), it->second.end());
	}
	delete right;
	return left;
}

ModelImpl::SynchronizedCommands *ModelImpl::flattenRespar(const System &system, set<string> &mentioned) {
	SynchronizedCommands *left = flatten(*system.left, mentioned);
	SynchronizedCommands *right = flatten(*system.right, mentioned);
	for (SynchronizedCommands::iterator it = right->begin(); it != right->end(); it++) {
		const Action &action = it->first;
		const Commands &cmds = it->second;
		if (system.actions.end() == system.actions.find(action)) {
			(*left)[action].insert((*left)[action].end(), cmds.begin(), cmds.end());
		} else {
			const Commands cmdsLeft = left->find(action)->second;
			left->erase(action);
			for (Commands::const_iterator leftIt = cmdsLeft.begin(); leftIt != cmdsLeft.end(); leftIt++) {
				for (Commands::const_iterator rightIt = cmds.begin(); rightIt != cmds.end(); rightIt++) {
					boost::shared_ptr<Command> newCmd((*(**leftIt).impl) * (*(**rightIt).impl));
					(*left)[action].push_back(newCmd);
				}
			}
		}
	}
	delete right;
	return left;
}

ModelImpl::SynchronizedCommands *ModelImpl::flatten(const System &system, set<string> &mentioned) {
	switch (system.type) {
	case System::IdentifierSystem:
		return flattenIdentifier(system, mentioned);
	case System::RenameSystem:
		return flattenRename(system, mentioned);
	case System::HideSystem:
		return flattenHide(system, mentioned);
	case System::AlparSystem:
		return flattenAlpar(system, mentioned);
	case System::AsparSystem:
		return flattenAspar(system, mentioned);
	case System::ResparSystem:
		return flattenRespar(system, mentioned);
	}
	return NULL;
}

void ModelImpl::flatten() {
	if (NULL == system) {
		createDefaultSystem();
	}

	set<string> mentioned;
	SynchronizedCommands *cmd = flatten(*system, mentioned);

	for (Modules::const_iterator it = modules.begin(); it != modules.end(); it++) {
		const string &name((*it)->impl->name);
		if (mentioned.end() == mentioned.find(name)) {
			throw prismparser_error("module \"" + name + "\" not mentioned in "
					"system definition");
		}
	}

	for (SynchronizedCommands::const_iterator it = cmd->begin(); it != cmd->end(); it++) {
		const Commands &c(it->second);
		for (Commands::const_iterator it2 = c.begin(); it2 != c.end(); it2++) {
			guarded_transitions.push_back(*it2);
		}
	}

	delete cmd;
}

void ModelImpl::ctmcDivideCommands() {
	assert(CTMC == model_type);
	Commands new_commands;
	for (unsigned cmdNr = 0; cmdNr < guarded_transitions.size(); cmdNr++) {
		const Command &cmd(*guarded_transitions[cmdNr]);
		const Expr &guard(cmd.getGuard());
		for (unsigned altNr = 0; altNr < cmd.getNumAlternatives(); altNr++) {
			const Alternative &alt(cmd.getAlternative(altNr));
			boost::shared_ptr<Command> c_ptr(new Command());
			c_ptr->impl->setGuard(guard);
			c_ptr->impl->setAction(cmd.getAction());
			Alternative *new_alt = alt.impl->clone();
			c_ptr->impl->addAlternative(new_alt);
			new_commands.push_back(c_ptr);
		}
	}
	guarded_transitions = new_commands;
}

// Notice: to be called after flatten().
void ModelImpl::fixDeadlocks() {
	Expr sinkGuard(Expr::trueExpr());
	for (unsigned gt_nr = 0; gt_nr < guarded_transitions.size(); gt_nr++) {
		Command &gt = *((guarded_transitions)[gt_nr]);
		const Expr &guard = gt.getGuard();
		sinkGuard = Expr::andExpr(sinkGuard, Expr::notExpr(guard));
	}

	Command *sinkCommand = new Command();
	sinkCommand->impl->setGuard(sinkGuard);
	sinkCommand->impl->setAction("");
	Alternative *sinkAlternative = new Alternative();
	sinkAlternative->impl->setWeight(Expr::ratExpr(1, 1));
	sinkCommand->impl->addAlternative(sinkAlternative);
	guarded_transitions.push_back(boost::shared_ptr<Command>(sinkCommand));
}

const Expr &ModelImpl::getInitial() const {
	return init;
}

const Expr &ModelImpl::getDefaultInitialValue(const Expr & var) const {
	ExprHashMap<Expr>::const_iterator it = default_initial_value.find(var);
	if (it != default_initial_value.end()) {
		return it->second;
	} else {
		static Expr dummy;
		return dummy;
	}
}

unsigned ModelImpl::getNumCommands() const {
	return guarded_transitions.size();
}

const Expr &ModelImpl::getStateReward(unsigned structure) const {
	return rewards.find(structure)->second.getStateRewardExpr();
}

const Expr &ModelImpl::getTransReward(unsigned structure, const string &action) const {
	return rewards.find(structure)->second.getTransRewardExpr(action);
}

const Command &ModelImpl::getCommand(unsigned commandNr) const {
	assert(commandNr < guarded_transitions.size());
	return *guarded_transitions[commandNr];
}

const vector<Expr> &ModelImpl::getUserPreds() const {
	return user_predicates;
}

const Property &ModelImpl::getProperty(unsigned propNr) const {
	assert(propNr < properties->size());
	return *(*properties)[propNr];
}

const vector<Expr> &ModelImpl::getInvar() const {
	return invariants;
}

bool ModelImpl::isParameterVariable(const Expr & var) const {
	return parameter_variables.find(var) != parameter_variables.end();
}

ModelType ModelImpl::getModelType() const {
	return model_type;
}

void ModelImpl::setDefaultInitialValue(const Expr & var, const Expr & val) {
	default_initial_value[var] = val;
}

void ModelImpl::setModelType(ModelType mt) {
	model_type = mt;
}

void ModelImpl::setAsParameter(const Expr &var) {
	parameter_variables.insert(var);
}

unsigned ModelImpl::getNumVariables() const {
	return var_names.size();
}

const Expr &ModelImpl::getVariable(unsigned varNr) const {
	assert(varNr < variables.size());
	return variables.find(var_names[varNr])->second;
}

const Expr &ModelImpl::getVariable(const string &name) const {
	std::tr1::unordered_map<string, Expr>::const_iterator it = variables.find(name);
	assert(variables.end() != it);
	return it->second;
}

const string &ModuleImpl::toString() const {
	if ("" == str) {
		for (SynchronizedCommands::const_iterator it = sync_guarded_transitions.begin(); it != sync_guarded_transitions.end();
				++it) {
			const Commands & gts = it->second;
			for (Commands::const_iterator it = gts.begin(); it != gts.end(); it++) {
				str += (*it)->toString();
			}
		}
		str += "\n";
	}

	return str;
}

void ModuleImpl::addCommand(Command *c) {
	boost::shared_ptr<Command> c_ptr(c);
	string a(c->getAction());
	sync_guarded_transitions[a].push_back(c_ptr);
}

ModuleImpl::ModuleImpl(const string &__name) {
	name = __name;
	refs = 0;
}

ModuleImpl::ModuleImpl() {
	refs = 0;
}

const string &ModuleImpl::getName() const {
	return name;
}

void CommandImpl::setGuard(const Expr & __g) {
	g = __g;
}

void CommandImpl::setAction(const string & __action) {
	action = __action;
}

CommandImpl::CommandImpl() {
	refs = 0;
	alternatives = new Alternatives();
}

CommandImpl::~CommandImpl() {
	delete alternatives;
}

/*********************************************************************/
// add an Assignment to the guarded transition.
/*********************************************************************/
void CommandImpl::addAlternative(Alternative *a) {
	assert(a);
	boost::shared_ptr<Alternative> a_ptr(a);
	alternatives->push_back(a_ptr);
	support.insert(a->impl->getSupport().begin(), a->impl->getSupport().end());
}

/*********************************************************************/
// Write the guarded transition to a string.
/*********************************************************************/
const string &CommandImpl::toString() const {
	if ("" == str) {
		str += "[" + action + "] ";
		str += g.toString() + " -> ";
		for (Alternatives::const_iterator it = alternatives->begin(); it != alternatives->end(); it++) {
			assert(*it);
			if (it != alternatives->begin())
				str += "\n + ";
			str += (*it)->toString();
		}
		str += ";\n";
	}

	return str;
}

/*********************************************************************/
//Warning: may also return 0
/*********************************************************************/
Command *CommandImpl::operator*(const CommandImpl &gt) const {
	//go through all alternatives, compose them and multiply their probabilities

	Command *ng = new Command();
	ng->impl->g = Expr::andExpr(g, gt.g);

	ng->impl->action = action;
	for (Alternatives::const_iterator it1 = alternatives->begin(); it1 != alternatives->end(); ++it1) {
		for (Alternatives::const_iterator it2 = gt.alternatives->begin(); it2 != gt.alternatives->end(); ++it2) {
			assert(*it1);
			assert(*it2);
			ng->impl->addAlternative(*(**it1).impl * *(**it2).impl);
		}
	}

	return ng;
}

/*********************************************************************/
//Warning: may also return 0
/*********************************************************************/
Command *CommandImpl::operator+(const CommandImpl &gt) const {
	//go through all alternatives, compose them and multiply their probabilities

	Command *ng = new Command();
	ng->impl->g = Expr::andExpr(g, gt.g);

	ng->impl->action = action;
	for (Alternatives::const_iterator it1 = alternatives->begin(); it1 != alternatives->end(); ++it1) {
		assert(*it1);
		ng->impl->addAlternative((*it1).get());
	}
	for (Alternatives::const_iterator it2 = gt.alternatives->begin(); it2 != gt.alternatives->end(); ++it2) {
		assert(*it2);
		ng->impl->addAlternative((*it2).get());
	}

	return ng;
}

/*! \brief getter for Command::guard */
const Expr &CommandImpl::getGuard() const {
	return g;
}

/*! \brief getter for Command::action */
const string &CommandImpl::getAction() const {
	return action;
}

const Alternative &CommandImpl::getAlternative(unsigned altNr) const {
	assert(altNr < alternatives->size());
	return *(*alternatives)[altNr];
}

unsigned CommandImpl::getNumAlternatives() const {
	return alternatives->size();
}

AlternativeImpl::AlternativeImpl() {
	refs = 0;
}

AlternativeImpl::~AlternativeImpl() {
}

void AlternativeImpl::Assign(const Expr &left, const Expr &right) {
	if (left.isNull() || right.isNull()) {
		throw prismparser_error("Assignment::Assign: assignment to Null " + left.toString() + " -> " + right.toString());
	}
	if (map.count(left) > 0) {
		throw prismparser_error(
				"Assignment::Assign: Conflicting assignment" + left.toString() + " -> { " + right.toString() + ","
						+ map[left].toString() + "}");
	}
	map.insert(make_pair(left, right));
	support.insert(left);
}

const string &AlternativeImpl::toString() const {
	if ("" == str) {
		str += weight.toString() + " : ";

		for (Map::const_iterator i = map.begin(); i != map.end(); ++i) {
			if (i != map.begin()) {
				str += " & ";
			}
			str += " ( " + (i->first).toString() + "' = " + (i->second).toString() + " ) ";
		}
	}

	return str;
}

Alternative* AlternativeImpl::clone() const {
	Alternative* a = new Alternative();
	a->impl->weight = weight;
	a->impl->map = map;
	a->impl->support = support;
	return a;
}

/*********************************************************************/
//return a pointer to an assignment
//containing
/*********************************************************************/
Alternative* AlternativeImpl::operator*(const AlternativeImpl& ass) const {
	Alternative *a = clone();

	a->impl->weight = Expr::multExpr(a->impl->weight, ass.weight);
	for (Map::const_iterator i = ass.map.begin(); i != ass.map.end(); i++) {
		a->impl->Assign(i->first, i->second);
	}

	return a;
}

Expr AlternativeImpl::operator()(const Expr& e) const {
	return e.substExpr(map);
}

void AlternativeImpl::setWeight(const Expr& weight_) {
	weight = weight_.simplify();
}

const Expr &AlternativeImpl::getWeight() const {
	return weight;
}

const ExprHashSet &AlternativeImpl::getSupport() const {
	return support;
}

const Expr &AlternativeImpl::getAssgnToVar(const Expr &var) const {
	assert(var.isVar());
	if (map.end() == map.find(var)) {
		return var;
	}
	return map.find(var)->second;
}

Model::Model() {
	impl = new ModelImpl();
	impl->refs++;
}

Model::Model(const Model &m) {
	impl = m.impl;
	impl->refs++;
}

Model::~Model() {
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
}

Model &Model::operator=(const Model &m) {
	m.impl->refs++;
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
	impl = m.impl;

	return *this;
}

unsigned Model::getNumProperties() const {
	return impl->getNumProperties();
}

const string &Model::toString() const {
	return impl->toString();
}

const Module &Model::getModule(const string &moduleName) const {
	return impl->getModule(moduleName);
}

const Expr &Model::getStateReward(unsigned structure) const {
	return impl->getStateReward(structure);
}

const Expr &Model::getTransReward(unsigned structure, const string &action) const {
	return impl->getTransReward(structure, action);
}

void Model::flatten() {
	impl->flatten();
}

void Model::ctmcDivideCommands() {
	impl->ctmcDivideCommands();
}

void Model::fixDeadlocks() {
	impl->fixDeadlocks();
}

const Expr &Model::getInitial() const {
	return impl->getInitial();
}

/*! \brief setter for initial default value */
const Expr &Model::getDefaultInitialValue(const Expr &var) const {
	return impl->getDefaultInitialValue(var);
}

unsigned Model::getNumCommands() const {
	return impl->getNumCommands();
}

const Command &Model::getCommand(unsigned commandNr) const {
	return impl->getCommand(commandNr);
}

const vector<Expr> &Model::getUserPreds() const {
	return impl->getUserPreds();
}

const Property &Model::getProperty(unsigned propNr) const {
	return impl->getProperty(propNr);
}

const vector<Expr> &Model::getInvar() const {
	return impl->getInvar();
}

bool Model::isParameterVariable(const Expr &var) const {
	return impl->isParameterVariable(var);
}

ModelType Model::getModelType() const {
	return impl->getModelType();
}

unsigned Model::getNumVariables() const {
	return impl->var_names.size();
}

const Expr &Model::getVariable(unsigned varNr) const {
	return impl->getVariable(varNr);
}

const Expr &Model::getVariable(const string &name) const {
	return impl->getVariable(name);
}

const string &Module::toString() const {
	return impl->toString();
}

Module::Module() {
	impl = new ModuleImpl();
	impl->refs++;
}

Module::Module(const Module &m) {
	impl = m.impl;
	impl->refs++;
}

Module::~Module() {
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
}

Module &Module::operator=(const Module &m) {
	m.impl->refs++;
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
	impl = m.impl;

	return *this;
}

const string &Module::getName() const {
	return impl->getName();
}

Command::Command() {
	impl = new CommandImpl();
	impl->refs++;
}

Command::Command(const Command &m) {
	impl = m.impl;
	impl->refs++;
}

Command::~Command() {
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
}

Command &Command::operator=(const Command &m) {
	m.impl->refs++;
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
	impl = m.impl;

	return *this;
}

const string &Command::toString() const {
	return impl->toString();
}

const Expr &Command::getGuard() const {
	return impl->getGuard();
}

const string &Command::getAction() const {
	return impl->getAction();
}

const Alternative &Command::getAlternative(unsigned altNr) const {
	return impl->getAlternative(altNr);
}

unsigned Command::getNumAlternatives() const {
	return impl->getNumAlternatives();
}

Alternative::Alternative() {
	impl = new AlternativeImpl();
	impl->refs++;
}

Alternative::Alternative(const Alternative &m) {
	impl = m.impl;
	impl->refs++;
}

Alternative::~Alternative() {
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
}

Alternative &Alternative::operator=(const Alternative &m) {
	m.impl->refs++;
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
	impl = m.impl;

	return *this;
}

const string &Alternative::toString() const {
	return impl->toString();
}

const Expr &Alternative::getWeight() const {
	return impl->getWeight();
}

const Expr &Alternative::getAssgnToVar(const Expr &var) const {
	return impl->getAssgnToVar(var);
}

ostream &operator<<(ostream &os, const Model &model) {
	os << model.toString();
	return os;
}

ostream &operator<<(ostream &os, const Module &module) {
	os << module.toString();
	return os;
}

ostream &operator<<(ostream &os, const Alternative &alternative) {
	os << alternative.toString();
	return os;
}

ostream &operator<<(ostream &os, const Command &command) {
	os << command.toString();
	return os;
}
}
