/*
 * This file is part of a parser for an extension of the PRISM language.
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The parser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the program this parser part of.
 * If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2007-2010 Bjoern Wachter (Bjoern.Wachter@comlab.ox.ac.uk)
 * Copyright 2011-2012 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#ifndef MODEL_IMPL_H
#define MODEL_IMPL_H

#include <string>
#include <set>
#include <map>
#include <tr1/unordered_map>
#include <tr1/unordered_set>
#include <boost/shared_ptr.hpp>
#include "Expr.h"
#include "Reward.h"

namespace prismparser {
class Model;
class Module;
class Command;
class Alternative;
class System;

class Properties: public std::vector<boost::shared_ptr<Property> > {
};

class ModelImpl {
private:
	friend class Model;

	ModelImpl();
	~ModelImpl();
	const Module &getModule(const std::string &) const;
	const Expr &getInitial() const;
	const Expr &getDefaultInitialValue(const Expr &) const;
	unsigned getNumCommands() const;
	const Command &getCommand(unsigned) const;
	const std::vector<Expr> &getUserPreds() const;
	unsigned getNumProperties() const;
	const Property &getProperty(unsigned) const;
	const std::vector<Expr> &getInvar() const;
	bool isParameterVariable(const Expr &) const;
	ModelType getModelType() const;
	const std::string &toString() const;
	unsigned getNumVariables() const;
	const Expr &getVariable(unsigned) const;
	const Expr &getVariable(const std::string &) const;

	typedef std::set<std::string> Actions;
	typedef std::vector<boost::shared_ptr<Command> > Commands;
	typedef std::map<std::string, Commands> SynchronizedCommands;
	typedef std::vector<boost::shared_ptr<Module> > Modules;

	void flatten();
	SynchronizedCommands *flatten(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenIdentifier(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenRename(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenHide(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenAlpar(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenAspar(const System &, std::set<std::string> &);
	SynchronizedCommands *flattenRespar(const System &, std::set<std::string> &);
	void ctmcDivideCommands();
	void fixDeadlocks();

	const Expr &getStateReward(unsigned) const;
	const Expr &getTransReward(unsigned, const std::string &) const;

	void computeDefaultInitialValue();
	void setDefaultInitialValue(const Expr &, const Expr &);
	void setModelType(ModelType);
	void addModule(Module *);
	void addAction(const Action &);
	void setInitial(const Expr &);
	void addVariable(const Expr &);
	void addPredicate(const Expr &);
	void addProperty(Property *);
	void addInvariant(const Expr &);
	void setAsParameter(const Expr &);
	void createDefaultSystem();

	unsigned refs;
	std::tr1::unordered_map<std::string, Expr> variables;
	ExprHashSet parameter_variables;
	ModelType model_type;
	Expr init;
	Commands guarded_transitions;
	Properties *properties;
	Modules modules;
	std::vector<Expr> user_predicates;
	std::vector<Expr> invariants;
	ExprHashMap<Expr> default_initial_value;
	std::vector<std::string> var_names;
	Actions actions;
	mutable std::string str;
	System *system;
	typedef std::map<unsigned, Reward> Rewards;
	Rewards rewards;

	friend class PRISMParserImpl;
};

class ModuleImpl {
private:
	friend class ModelImpl;
	friend class Module;
	friend class Model;
	friend class PRISMParserImpl;

	ModuleImpl(const std::string &);
	ModuleImpl();
	void addCommand(Command *);
	const std::string &getName() const;
	const std::string &toString() const;
	typedef std::vector<boost::shared_ptr<Command> > Commands;
	typedef std::map<std::string, Commands> SynchronizedCommands;

	unsigned refs;
	SynchronizedCommands sync_guarded_transitions;
	std::tr1::unordered_map<std::string, Expr> local_vars;
	std::string name;
	mutable std::string str;
};

struct CommandImpl {
private:
	friend class Model;
	friend class ModelImpl;
	friend class AbsModel;
	friend class Command;
	friend class PRISMParserImpl;

	CommandImpl();
	~CommandImpl();
	const Expr &getGuard() const;
	const std::string &getAction() const;
	const Alternative &getAlternative(unsigned) const;
	unsigned getNumAlternatives() const;
	void setGuard(const Expr &);
	void setAction(const std::string &);
	void addAlternative(Alternative *);
	const std::string &toString() const;
	const ExprHashSet &getSupport() const;
	Expr WP(const Expr &e, unsigned i) const;

	typedef std::vector<boost::shared_ptr<Alternative> > Alternatives;

	Command *operator+(const CommandImpl &) const;
	Command *operator*(const CommandImpl &) const;

	unsigned refs;
	Action action;
	Expr g;
	Alternatives *alternatives;
	ExprHashSet support;
	mutable std::string str;
};

class AlternativeImpl {
private:
	friend class Alternative;
	friend class CommandImpl;
	friend class PRISMParserImpl;
	friend class ModelImpl;

	AlternativeImpl();
	~AlternativeImpl();
	const Expr &getAssgnToVar(const Expr &) const;
	const Expr &getWeight() const;
	const std::string &toString() const;
	void setWeight(const Expr &);
	Alternative* clone() const;
	void Assign(const Expr &, const Expr &);
	Alternative* operator*(const AlternativeImpl &) const;
	Expr operator()(const Expr &) const;
	typedef ExprHashMap<Expr> Map;
	const ExprHashSet& getSupport() const;

	unsigned refs;
	Expr weight;
	Map map;
	Map wp_cache;
	ExprHashSet support;
	mutable std::string str;
};
}

#endif
