/*
 * This file is part of a parser for an extension of the PRISM language.
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The parser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the program this parser part of.
 * If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2007-2010 Bjoern Wachter (Bjoern.Wachter@comlab.ox.ac.uk)
 * Copyright 2009-2012 Ernst Moritz Hahn (emh@cs.uni-saarland.de)
 */

#include <sstream>
#include <limits>
#include <cassert>
#include "Util.h"
#include "Expr.h"
#include "Property.h"
#include "PropertyImpl.h"

namespace prismparser {
using namespace std;

TimeType TimeBoundImpl::getType() const {
	if ((0 < t1) && (numeric_limits<double>::infinity() == t2)) {
		return GeTime;
	} else if ((0 == t1) && (numeric_limits<double>::infinity() > t2)) {
		return LeTime;
	} else if ((0 < t1) && (numeric_limits<double>::infinity() > t2)) {
		return IntervalTime;
	} else {
		return UnboundedTime;
	}
}

const string &TimeBoundImpl::toString() const {
	if ("" == str) {
		stringstream stream;
		stream << *this;
		str = stream.str();
	}
	return str;
}

double TimeBoundImpl::getLowerBound() const {
	return t1;
}

double TimeBoundImpl::getUpperBound() const {
	return t2;
}

bool TimeBoundImpl::hasUpperBound() const {
	return (numeric_limits<double>::infinity() != t2);
}

TimeBoundImpl::TimeBoundImpl() {
}

TimeBoundImpl::~TimeBoundImpl() {
}

ostream &operator<<(ostream &stream, const TimeBoundImpl &bound) {
	switch (bound.getType()) {
	case GeTime:
		stream << ">=" << bound.t1;
		break;
	case LeTime:
		stream << "<=" << bound.t2;
		break;
	case IntervalTime:
		stream << "[" << bound.t1 << "," << bound.t2 << "]";
		break;
	case UnboundedTime:
		stream << "[0,infty]";
		break;
	default:
		assert(false);
		break;
	}
	return stream;
}

ostream &operator<<(ostream &stream, const TimeBound &bound) {
	stream << *bound.impl;
	return stream;
}

const string &ProbBoundImpl::toString() const {
	if ("" == str) {
		stringstream stream;
		stream << *this;
		str = stream.str();
	}
	return str;
}

bool ProbBoundImpl::isMinOrMax() const {
	return minOrMax;
}

ProbBoundImpl::ProbBoundImpl() {
	type = DkBound;
	bound = NULL;
}

ProbBoundImpl::~ProbBoundImpl() {
	if (NULL != bound) {
		delete bound;
	}
}

bool ProbBoundImpl::isMin() const {
	return min;
}

const Expr &ProbBoundImpl::getBound() const {
	return *bound;
}

BoundType ProbBoundImpl::getType() const {
	return type;
}

ostream &operator<<(ostream &stream, const ProbBoundImpl &bound) {
	if (bound.minOrMax) {
		if (bound.min) {
			stream << "min";
		} else {
			stream << "max";
		}
	}

	switch (bound.type) {
	case GrBound:
		stream << ">";
		break;
	case GeBound:
		stream << ">=";
		break;
	case LtBound:
		stream << "<";
		break;
	case LeBound:
		stream << "<=";
		break;
	case DkBound:
		stream << "=?";
		break;
	default:
		assert(false);
		break;
	}

	if (DkBound != bound.type) {
		stream << *bound.bound;
	}
	return stream;
}

ostream &operator<<(ostream &stream, const ProbBound &bound) {
	stream << *bound.impl;
	return stream;
}

FilterImpl::FilterImpl() {
	expr = NULL;
}

FilterImpl::~FilterImpl() {
	if (NULL != expr) {
		delete expr;
	}
}

const Expr &FilterImpl::getExpr() const {
	return *expr;
}

FilterType FilterImpl::getType() const {
	return type;
}

PropertyImpl::PropertyImpl() {
	filter = NULL;
	rewStructNr = numeric_limits<unsigned>::max();
	probBound = NULL;
	doubleTime = -1.0;
	timeBound = NULL;
	expr = NULL;
	refs = 0;
}

PropertyImpl::~PropertyImpl() {
	if (NULL != filter) {
		delete filter;
	}
	if (NULL != probBound) {
		delete probBound;
	}
	if (NULL != timeBound) {
		delete timeBound;
	}
	if (NULL != expr) {
		delete expr;
	}
	for (unsigned childNr = 0; childNr < children.size(); childNr++) {
		delete children[childNr];
	}
}

const ProbBound &PropertyImpl::getProbBound() const {
	return *probBound;
}

PropType PropertyImpl::getType() const {
	return type;
}

unsigned PropertyImpl::arity() const {
	return children.size();
}

const Property &PropertyImpl::operator[](unsigned childNr) const {
	assert(childNr < children.size());
	return *children[childNr];
}

const TimeBound &PropertyImpl::getTimeBound() const {
	assert(NULL != timeBound);
	return *timeBound;
}

const Filter &PropertyImpl::getFilter() const {
	assert(NULL != filter);
	return *filter;
}

bool PropertyImpl::hasFilter() const {
	return (NULL != filter);
}

unsigned PropertyImpl::getRewStructNr() const {
	assert(numeric_limits<unsigned>::max() != rewStructNr);
	return rewStructNr;
}

const double PropertyImpl::getTime() const {
	assert(-1.0 != doubleTime);
	return doubleTime;
}

const Expr &PropertyImpl::getExpr() const {
	assert(NULL != expr);
	return *expr;
}

const string &PropertyImpl::toString() const {
	if ("" == str) {
		stringstream stream;
		stream << *this;
		str = stream.str();
	}
	return str;
}

const string &TimeBound::toString() const {
	return impl->toString();
}

TimeBound::TimeBound() {
	impl = new TimeBoundImpl();
}

TimeBound::~TimeBound() {
	delete impl;
}

TimeType TimeBound::getType() const {
	return impl->getType();
}

double TimeBound::getLowerBound() const {
	return impl->getLowerBound();
}

double TimeBound::getUpperBound() const {
	return impl->getUpperBound();
}

bool TimeBound::hasUpperBound() const {
	return impl->hasUpperBound();
}

const string &ProbBound::toString() const {
	return impl->toString();
}

bool ProbBound::isMinOrMax() const {
	return impl->isMinOrMax();
}

ProbBound::ProbBound() {
	impl = new ProbBoundImpl();
	impl->type = DkBound;
}

ProbBound::~ProbBound() {
	delete impl;
}

bool ProbBound::isMin() const {
	return impl->isMin();
}

BoundType ProbBound::getType() const {
	return impl->getType();
}

const Expr &ProbBound::getBound() const {
	return impl->getBound();
}

Filter::Filter() {
	impl = new FilterImpl();
}

Filter::~Filter() {
	delete impl;
}

const Expr &Filter::getExpr() const {
	return impl->getExpr();
}

FilterType Filter::getType() const {
	return impl->getType();
}

Property::Property() {
	impl = new PropertyImpl();
	impl->refs++;
}

Property::Property(const Property &m) {
	impl = m.impl;
	impl->refs++;
}

Property::~Property() {
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
}

Property &Property::operator=(const Property &m) {
	m.impl->refs++;
	impl->refs--;
	if (0 == impl->refs) {
		delete impl;
	}
	impl = m.impl;

	return *this;
}

const ProbBound &Property::getProbBound() const {
	return impl->getProbBound();
}

PropType Property::getType() const {
	return impl->getType();
}

unsigned Property::arity() const {
	return impl->arity();
}

const Property &Property::operator[](unsigned childNr) const {
	return (*impl)[childNr];
}

const TimeBound &Property::getTimeBound() const {
	return impl->getTimeBound();
}

const Filter &Property::getFilter() const {
	return impl->getFilter();
}

bool Property::hasFilter() const {
	return impl->hasFilter();
}

unsigned Property::getRewStructNr() const {
	return impl->getRewStructNr();
}

const double Property::getTime() const {
	return impl->getTime();
}

const Expr &Property::getExpr() const {
	return impl->getExpr();
}

const string &Property::toString() const {
	return impl->toString();
}

ostream &operator<<(ostream &stream, const Property &prop) {
	stream << *prop.impl;
	return stream;
}

ostream &operator<<(ostream &stream, const PropertyImpl &prop) {
	if (NULL != prop.filter) {
		stream << "filter(";
		switch (prop.filter->impl->type) {
		case MinFilter:
			stream << "min";
			break;
		case MaxFilter:
			stream << "max";
			break;
		case CountFilter:
			stream << "count";
			break;
		case SumFilter:
			stream << "sum";
			break;
		case AvgFilter:
			stream << "avg";
			break;
		case FirstFilter:
			stream << "first";
			break;
		case RangeFilter:
			stream << "range";
			break;
		case ForallFilter:
			stream << "forall";
			break;
		case ExistsFilter:
			stream << "exists";
			break;
		case StateFilter:
			stream << "state";
			break;
		case ArgminFilter:
			stream << "argmin";
			break;
		case ArgmaxFilter:
			stream << "argmax";
			break;
		case PrintFilter:
			stream << "print";
			break;
		}
		stream << ", ";
	}

	switch (prop.type) {
	case ExprProp:
		stream << *prop.expr;
		break;
	case ReachRewProp:
		stream << "R{" << (prop.rewStructNr + 1) << "}" << *prop.probBound << " [ F " << *prop.children[0] << "]";
		break;
	case CumulRewProp: {
		stream << "R{" << (prop.rewStructNr + 1) << "}" << *prop.probBound << " [ C<=" << prop.doubleTime << " ]";
		break;
	}
	case InstRewProp:
		stream << "R{" << (prop.rewStructNr + 1) << "}" << *prop.probBound << " [ I=" << prop.doubleTime << "]";
		break;
	case SteadySRewProp:
		stream << "R{" << (prop.rewStructNr + 1) << "}";
		stream << *prop.probBound << " [ S ]";
		break;
	case SteadySProp:
		stream << "S" << *prop.probBound;
		stream << "[ " << *prop.children[0] << " ]";
		break;
	case NegProp:
		stream << "!" << *prop.children[0];
		break;
	case AndProp:
		stream << *prop.children[0] << " & " << *prop.children[1];
		break;
	case OrProp:
		stream << *prop.children[0] << " | " << *prop.children[1];
		break;
	case ImplProp:
		stream << *prop.children[0] << " => " << *prop.children[1];
		break;
	case QuantProp:
		stream << "P";
		stream << *prop.probBound;
		stream << " [ " << *prop.children[0] << " ]";
		break;
	case NextProp:
		stream << "X " << *prop.timeBound << " " << *prop.children[0];
		break;
	case UntilProp:
		if (ExprProp != prop.children[0]->getType()) {
			stream << "(";
		}
		stream << *prop.children[0] << " U";
		if (ExprProp != prop.children[0]->getType()) {
			stream << ")";
		}
		stream << *prop.timeBound << " ";
		if (ExprProp != prop.children[1]->getType()) {
			stream << "(";
		}
		stream << *prop.children[1];
		if (ExprProp != prop.children[1]->getType()) {
			stream << ")";
		}
	}
	if (NULL != prop.filter) {
		stream << ", ";
		stream << *prop.filter->impl->expr;
		stream << ")";
	}
	return stream;
}
}
