/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "ModelChecker.h"
#include "../io/IOWrapper.h"

#ifdef USE_PARAMETRIC
#include "../data/Parameters.h"
#include "../data/Rational.h"
#include "../data/Cancellator2.h"
using namespace parametric;
#endif

namespace scc_cex {

Logger mclogger = Logger::getInstance("ModelCheck");

ModelChecker::ModelChecker() {
	mGlobalGraph = NULL;
	doPreprocessing = true;
	mAuxInputNode = NULL;
	counter = 0;

	mclogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

ModelChecker::~ModelChecker() {
	LOG4CPLUS_TRACE(mclogger, "Will now get rid of the model checker");
	if (!mGlobalGraph->limitedFree)
		delete mGlobalGraph;
	mAuxInputNode = NULL;
}

void ModelChecker::disablePreprocessing(bool doDisable) {
	doPreprocessing = !doDisable;
}

void ModelChecker::setAuxInputNode(Vertex<PTYPE> *auxNode) {
	mAuxInputNode = auxNode;
}

PTYPE ModelChecker::getTargetResult(Vertex<PTYPE>* target) const {
	return getTargetResult(target->oindex);
}

PTYPE ModelChecker::getTargetResult(unsigned int trgIndex) const {
	//No. of input nodes should be one, but we won't enforce it...
	if (mGlobalGraph->inputVertices.size() == 1) {
		LOG4CPLUS_TRACE(mclogger, "Graph has one input, as expected (" << (*(mGlobalGraph->inputVertices[0])) << ")");
	} else if (mGlobalGraph->inputVertices.size() > 1) {
		LOG4CPLUS_TRACE(mclogger, "Graph has too many input vertices... (" << mGlobalGraph->inputVertices.size() << ")");
	} else {
		LOG4CPLUS_TRACE(mclogger, "Error: Graph doesn't have any input vertices...");
	}

	for (unsigned int i = 0; i < mGlobalGraph->inputVertices.size(); i++) {
		Vertex<PTYPE>* vtx = mGlobalGraph->inputVertices[i];
		for (unsigned int j = 0; j < vtx->outEdges.size(); j++) {
			if (vtx->outEdges[j]->state == ABSTRACT && vtx->outEdges[j]->target->oindex == trgIndex) {
				return vtx->outEdges[j]->prob;
			}
		}
	}

	return 0;
}

Graph<PTYPE>* ModelChecker::modelCheck_common() {
	Graph<PTYPE>& g = *mGlobalGraph;

	const clock_t t_start = clock();

	//Model Checking depending on chosen approach
	modelCheck_common(&g);

	const clock_t t_end = clock();

	//LOG4CPLUS_DEBUG(mclogger, "Input for dReal:");
	//LOG4CPLUS_DEBUG(mclogger, Parameters::getInstance()->convertToDRealInput());

	//Output
#ifdef USE_PARAMETRIC
	LOG4CPLUS_DEBUG(mclogger, Parameters::getInstance().toString());

	float cancelTime = ModelChecker::convertTime(Cancellator::getInstance().getCancelTime());
	LOG4CPLUS_INFO(mclogger, "Cancel time: " << cancelTime);
	log << "Cancel time:\t" << cancelTime << endl;

	float additionTime = ModelChecker::convertTime(Cancellator::getInstance().getAdditionTime());
	LOG4CPLUS_INFO(mclogger, "Addition (Factor) time: " << additionTime);
	log << "Addition (Factor) time:\t" << additionTime << endl;

	float commonCoefficientTime = ModelChecker::convertTime(Cancellator::getInstance().getCommonCoefficientTime());
	LOG4CPLUS_INFO(mclogger, "Common coefficient time: " << commonCoefficientTime);
	log << "Common coefficient time:\t" << commonCoefficientTime << endl;

	stringstream numPolynomials;
	numPolynomials << Parameters::getInstance().getNumberPolynomials();

#ifdef USE_POLY_MANAGEMENT
	numPolynomials << " + " << Parameters::getInstance().getNumberOldPolynomials();
#endif

	LOG4CPLUS_INFO(mclogger, "Polynomial count: " << numPolynomials.str());
	log << "Polynomial count:\t" << numPolynomials.str() << endl;
#endif

	float totalTime = ModelChecker::convertTime(t_end - t_start);
	LOG4CPLUS_INFO(mclogger, "Total time: " << totalTime);
	log << "Total time:\t" << totalTime << endl;

	return &g;
}

Graph<PTYPE>* ModelChecker::modelCheck(Graph<PTYPE>* graph) {
	mGlobalGraph = graph;
	mGlobalGraph->limitedFree = true;

	return modelCheck_common();
}

Graph<PTYPE>* ModelChecker::modelCheck(vtx_vector & vertexList, int& initialNode, vector<bool>& targetNodes,
		map<int, AtPropType> &vtxToPropsMap, unsigned int edgeCount) {
	/*if (mGlobalGraph!=NULL){
	 delete mGlobalGraph;
	 mGlobalGraph = NULL;
	 }*/

	//initialize the graph if necessary
	//if (mGlobalGraph == NULL) {
	log << "States:\t" << vertexList.size() << endl;
	log << "Transitions:\t" << edgeCount << endl;
	LOG4CPLUS_TRACE(mclogger, "initializing graph...");

	mGlobalGraph = new Graph<PTYPE>();
	Graph<PTYPE>& g = *mGlobalGraph;
	initGraph(vertexList, initialNode, targetNodes, g, vtxToPropsMap);

	LOG4CPLUS_TRACE(mclogger, "initialization finished");

	LOG4CPLUS_TRACE(mclogger, g);

	return modelCheck_common();
}

void ModelChecker::initGraph(vtx_vector & vertexList, unsigned int initialNode, vector<bool> & targetNodes, Graph<PTYPE> & g,
		map<int, AtPropType>& vtxToPropsMap) {
	assert(initialNode < vertexList.size());

	//doPreprocessing = false;

	LOG4CPLUS_TRACE(mclogger, vertexList.size() << " vertices given, initial node being " << initialNode);

	//remove all vertices from the graph g
	//that are not reachable from the initial vertex of g
	vector<Vertex<PTYPE>*> tmpList;
	tmpList.push_back(vertexList[initialNode]);
	vertexList[initialNode]->seen = true;
	for (unsigned int i = 0; i < tmpList.size(); i++) {
		for (unsigned int j = 0; j < tmpList[i]->outEdges.size(); j++) {
			Vertex<PTYPE>* v = tmpList[i]->outEdges[j]->target;
			if (!v->seen) {
				tmpList.push_back(v);
				v->seen = true;
			}
		}
	}
	int pos = 0;
	for (unsigned int i = 0; i < vertexList.size(); i++) {
		if (vertexList[i] != NULL) {
			if (!vertexList[i]->seen && i != initialNode && !targetNodes[i])
				delete vertexList[i];
			else {
				vertexList[i]->seen = false;
				vertexList[i]->g = &g;
				vertexList[pos++] = vertexList[i];
			}
		}
	}
	vertexList.resize(pos);

	vector<Vertex<PTYPE>*> toScale;

	//We iterate through all reachable vertices to determine their type and, if
	//feasible, remove vertices
	for (unsigned int i = 0; i < vertexList.size(); i++) {
		Vertex<PTYPE>* v = vertexList[i];

		int od = v->outEdges.size();
		int id = v->inEdges.size();

		LOG4CPLUS_TRACE(mclogger, (*v) << ": id = " << id << ", od = " << od << ": ");

		if (od > 0 || id > 0) {

			if (od == 0) {
				v->g = NULL;
				g.outputVertices.push_back(v);

				LOG4CPLUS_TRACE(mclogger, "found output vertex " << (*v));
			} else if (id == 0) {

				assert(v->oindex == initialNode);
				g.allVertices.push_back(v);
				g.inputVertices.push_back(v);
				v->isInputVertex = true;
				v->isConcretized = false;

				LOG4CPLUS_TRACE(mclogger, "found input vertex " << (*v));
			} else /* vtx has both incoming and outgoing edges */{

				//eliminate vertices with a single input and
				//a single output edge
				if (od == 1 && id == 1 && doPreprocessing) {
					//       if (false) {

					LOG4CPLUS_DEBUG(mclogger, "eliminating vertex " << (*v));

					Vertex<PTYPE>* s = v->inEdges[0]->source;
					Vertex<PTYPE>* t = v->outEdges[0]->target;
#ifdef DEBUGGING
					assert(s != v);
					assert(v->oindex != initialNode);
#ifdef USE_PARAMETRIC
					Parameters::getInstance().checkConstraintGreaterZero(v->inEdges[0]->prob);
					Parameters::getInstance().checkConstraintGreaterZero(v->outEdges[0]->prob);
#endif
#endif
					PTYPE prob = v->inEdges[0]->prob * v->outEdges[0]->prob;

					if (s == t) {
						//Scale other edges
						for (unsigned int j = 0; j < s->outEdges.size(); j++) {
							s->outEdges[j]->prob /= prob;
						}
						delete v;
					} else {
						Edge<PTYPE>* e = NULL;
						for (unsigned int k = 0; k < s->outEdges.size(); k++) {
							if (s->outEdges[k]->target == t) {
								e = s->outEdges[k];
								break;
							}
						}
						if (e == NULL) {
							new Edge<PTYPE>(s, t, prob);
						} else {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
							Parameters::getInstance().checkConstraintGreaterZero(e->prob);
#endif
#endif
							e->prob += prob;
						}
						delete v;
					}
				} else /* multiple incoming/outgoing or no preprocessing */{

					g.allVertices.push_back(v);

					LOG4CPLUS_TRACE(mclogger, "keeping vertex " << (*v));
				}
			}
		}
		//v has neither incoming nor outgoing transitions and is no special node
		//--> unreachable node that we can safely remove
		//TODO: Labels will also play a role here
		else if (v->oindex != initialNode && !targetNodes[v->oindex] && doPreprocessing) {
			LOG4CPLUS_TRACE(mclogger, "removing vertex " << (*v));
			delete v;
		}
		//last case: no edges to/from v, but special vertex or no preprocessing --> must keep it
		else {
			LOG4CPLUS_TRACE(mclogger, "special vertex " << (*v));

			if (v->oindex == initialNode) {
				g.inputVertices.push_back(v);
				v->isInputVertex = true;
				v->isConcretized = false;
				g.allVertices.push_back(v);
			} else {
				g.outputVertices.push_back(v);
				v->g = NULL;
			}
		}
	}

	vertexList.clear();
}

} /* namespace scc_cex */

/* -------------- functions for external callers --------------- */

int callModelCheckerWithFile(const string fileName) {
	//scc_cex::IOWrapper wrapper(fileName, IOFL_INPUT_IS_FILE | IOFL_DTMC_ONLY | IOFL_EXTERNAL_CALLER);
	scc_cex::IOWrapper wrapper(fileName, 8);
	wrapper.run();
	return 0;
}

int callModelCheckerWithString(const string dataString) {
	//scc_cex::IOWrapper wrapper(dataString, IOFL_INPUT_IS_DATA | IOFL_DTMC_ONLY | IOFL_EXTERNAL_CALLER);
	scc_cex::IOWrapper wrapper(dataString, 9);
	wrapper.run();
	return 0;
}
