/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef MODELCHECKER_H_
#define MODELCHECKER_H_

#include <vector>
#include <map>
#include <string>
#include <time.h>

#include "../defines.h"
#include "../typedefs.h"
#include "../data/GraphTypes.h"

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

using namespace log4cplus;
using namespace std;

namespace scc_cex {

extern Logger mclogger;

/**
 * Abstract class for all model checking algorithms
 */
class ModelChecker {
public:
	ModelChecker();
	virtual ~ModelChecker();

	/* configuration functions */
	void setAuxInputNode(Vertex<PTYPE> *auxNode);
	void disablePreprocessing(bool doDisable);

	/* control functions */
	PTYPE getTargetResult(Vertex<PTYPE>* target) const;PTYPE getTargetResult(unsigned int trgIndex) const;
	Graph<PTYPE>* modelCheck(vtx_vector & vertexList, int& initialNode, vector<bool>& targetNodes,
			map<int, AtPropType>& vtxToPropsMap, unsigned int edgeCount);
	Graph<PTYPE>* modelCheck(Graph<PTYPE>* graph);
	void initGraph(vtx_vector & vertexList, unsigned int initialNode, std::vector<bool> & targetNodes, Graph<PTYPE> & g,
				std::map<int, AtPropType>& vtxToPropsMap);


	string getLog() const {
		return log.str();
	}

	static float convertTime(clock_t time) {
		return (float) ((float) time / (float) CLOCKS_PER_SEC);
	}

	Graph<PTYPE> *mGlobalGraph;
protected:
	virtual void modelCheck_common(Graph<PTYPE>* g) = 0;
	Graph<PTYPE>* modelCheck_common();

	Vertex<PTYPE> *mAuxInputNode;
	bool doPreprocessing;
	unsigned int counter;
	stringstream log;
};

//For JNI calls
int callModelCheckerWithFile(const string fileName);
int callModelCheckerWithString(const string dataString);

} /* namespace scc_cex */
#endif /* MODELCHECKER_H_ */
