/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <queue>

#include "SCC_MC.h"
#include "../io/Configuration.h"

#ifdef USE_PARAMETRIC
#include "../data/Parameters.h"
#endif

namespace scc_cex {

void SccMc::modelCheck_common(Graph<PTYPE>* g) {
	const clock_t t_scc_start = clock();

	//recursively find SCCs
	//makes the SCCs available as subgraphs of g,
	//the SCCs in the SCCs as subgraphs of the subgraphs etc.
	assert(g->subgraphs.size() == 0);
	assert(g->inputVertices.size() == 1);
	strongComponents(g);

	const clock_t t_scc_end = clock();

	//abstraction of SCCs
	LOG4CPLUS_TRACE(mclogger, "### starting abstraction... ###");

	for (unsigned int i = 0; i < g->subgraphs.size(); i++) {
		abstraction(g->subgraphs[i]);
	}

	LOG4CPLUS_TRACE(mclogger, "### time for the final reduction... ###");
	reduceGraph(g);

	const clock_t t_end = clock();

	LOG4CPLUS_INFO(mclogger, "MODEL CHECKING RESULT:\n" << *g << "(" << counter << " SCCs with multiple input nodes handled)");

	if (mAuxInputNode != NULL) {
		if (mAuxInputNode->g != NULL) {
			assert(mAuxInputNode->g->state == ABSTRACT);
			mAuxInputNode->g->state = CONCRETE;
		} else {
			LOG4CPLUS_WARN(mclogger, "aux. node not null (" << (*mAuxInputNode) << "), but no graph set");
		}
	}

	//Output
	float sccDetectionTime = ModelChecker::convertTime(t_scc_end - t_scc_start);
	LOG4CPLUS_INFO(mclogger, "SCC detection time: " << sccDetectionTime);
	log << "SCC detection time:\t" << sccDetectionTime << endl;

	float sccEliminationTime = ModelChecker::convertTime(t_end - t_scc_end);
	LOG4CPLUS_INFO(mclogger, "SCC elimination time: " << sccEliminationTime);
	log << "SCC elimination time:\t" << sccEliminationTime << endl;
}

void SccMc::computeWeight(Vertex<PTYPE> *in, Vertex<PTYPE> *out) const {

	LOG4CPLUS_TRACE(mclogger,
			"comp. weight " << (*in) << " -> " << (*out) << ", trg having " << out->inEdges.size() << " in edges");

	out->weight = 0.0;
	for (unsigned int i = 0; i < out->inEdges.size(); i++) {
		Vertex<PTYPE>* v = out->inEdges[i]->source;
		if (v->g->state == out->inEdges[i]->state && (v->g->parent == NULL || v->g->parent->state == CONCRETE)) {
			Graph<PTYPE>* g2 = v->g;
			while (g2 != NULL && g2 != in->g)
				g2 = g2->parent;
			if (g2 == in->g) {
				if (v->weight == -1.0) {
					computeWeight(in, v);
				}
				out->weight += v->weight * out->inEdges[i]->prob;
			}
		}
	}
	LOG4CPLUS_TRACE(mclogger, "result: " << (*out));
}

//TODO: Return to the algorithm that doesn't compute loops
void SccMc::abstraction(Graph<PTYPE>* g) {
	//JK: Maybe a bit too much DEBUGGING here... Might remove later
	LOG4CPLUS_TRACE(mclogger, "--next level of abstraction--");

	for (unsigned int i = 0; i < g->subgraphs.size(); i++)
		abstraction(g->subgraphs[i]);

	//Simple cases: Only one input and/or output vertex
	if (g->inputVertices.size() == 1 || g->outputVertices.size() == 1) {
		//One call to reduce graph suffices; the abstract edges are added there
		reduceGraph(g);
	}
	//More interesting case: Multiple input and output nodes
	else {

		if (DEFAULT_LOG_LEVEL <= TRACE_LOG_LEVEL) {
			counter++;

			LOG4CPLUS_TRACE(mclogger, "abstract: reducing SCC " << g->graph_id << " with multiple input/multiple output");
			stringstream ss;
			ss << "input nodes are: ";

			for (unsigned int i = 0; i < g->inputVertices.size(); i++) {
				ss << g->inputVertices[i]->oindex;
				ss << " ";
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
		}

		//Collect all vertices of the SCC
		vector<Vertex<PTYPE>*> vertices;

		//First output vertices
		for (unsigned int i = 0; i < g->outputVertices.size(); i++) {
			Vertex<PTYPE>* v = g->outputVertices[i];
			assert(i == vertices.size());
			v->dfs = i;
			vertices.push_back(v);
			v->seen = true;
		}
		//Then input vertices
		for (unsigned int i = 0; i < g->inputVertices.size(); i++) {
			Vertex<PTYPE>* v = g->inputVertices[i];
			v->seen = true;
			v->dfs = vertices.size();
			vertices.push_back(v);
		}

		//Needed for loop computations
		const unsigned int numOV = g->outputVertices.size();
		const unsigned int numIV = g->inputVertices.size();

		//Then intermediate vertices in the same SCC
		for (unsigned int i = numOV; i < vertices.size(); i++) {
			Vertex<PTYPE>* source = vertices[i];
			for (unsigned int j = 0; j < source->outEdges.size(); j++) {
				Edge<PTYPE>* e = source->outEdges[j];
				if (e->state == source->g->state) {
					Vertex<PTYPE>* target = e->target;
					if (!target->seen) {
						target->seen = true;
						target->dfs = vertices.size();
						vertices.push_back(target);
					}
				}
			}
		}
		for (unsigned int i = 0; i < vertices.size(); i++) {
			vertices[i]->seen = false;
		}

		const unsigned int nodeCountOutput = vertices.size();
		const unsigned int nodeCount = nodeCountOutput - numOV;
		//One row for each non-output node (i.e. all possible edge sources),
		//One column for each node (i.e. all possible target nodes of the scc's edges)
		LOG4CPLUS_TRACE(mclogger, "Matrix size: " << nodeCount << "x" << nodeCountOutput);
		//TODO use sparse matrix here
		PTYPE prob[nodeCount][nodeCountOutput];

		//Initialize matrix
		for (unsigned int j1 = 0; j1 < nodeCount; j1++) {
			for (unsigned int j2 = 0; j2 < nodeCountOutput; j2++) {
				prob[j1][j2] = 0;
			}

			//Fill with probabilities of edges
			Vertex<PTYPE>* source = vertices[j1 + numOV];
			assert(source->dfs == (int ) (j1 + numOV));
			for (unsigned int j2 = 0; j2 < source->outEdges.size(); j2++) {
				Edge<PTYPE>* e = source->outEdges[j2];
				if (e->state == source->g->state) {
					prob[j1][e->target->dfs] = e->prob;
				}
			}
		}

		if (DEFAULT_LOG_LEVEL <= TRACE_LOG_LEVEL) {
			stringstream ss;
			ss << "initial matrix:\n";
			//Column header
			ss << "         ";
			for (unsigned int j2 = 0; j2 < nodeCountOutput; j2++) {
				stringstream tmp;
				tmp << "p" << vertices[j2]->oindex;
				ss << setw(6) << left << tmp.str();
				ss << " | ";
			}
			ss << "\n";

			//Probability table
			for (unsigned int j1 = 0; j1 < nodeCount; j1++) {
				stringstream tmp;
				tmp << "p" << vertices[j1 + numOV]->oindex << " = ";
				ss << setw(9) << left << tmp.str();
				PTYPE sum = 0;
				for (unsigned int j2 = 0; j2 < nodeCountOutput; j2++) {
					ss << setw(6) << left << prob[j1][j2] << " | ";
					sum += prob[j1][j2];
				}
				ss << "\n";
				assert(sum == 1);
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
			LOG4CPLUS_TRACE(mclogger, "now computing final matrix...");
		}

		//We eliminate all intermediate nodes, i.e. nodes that are neither input nor output nodes
		//For each intermediate node...
		for (int currNode = nodeCount - 1; currNode >= (int) numIV; currNode--) {
			const int currNodeCol = currNode + numOV;
			//...and for each node with an edge to that intermediate node,
			//   i.e. with a non-zero entry in column col...

			//TODO only iterate over predecessors? Pro: no zero entries, contra: possible more edges which are not part of the scc
			for (unsigned int srcNode = 0; srcNode < nodeCount; srcNode++) {
				if ((int) srcNode == currNode) {
					continue; //Can't eliminate self-loops, of course!
				}

				PTYPE weight = 1 - prob[currNode][currNodeCol];

				//...get the probability of taking a transition from node srcNode to currNode...
				PTYPE fac = prob[srcNode][currNodeCol];
				if (fac != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
					parametric::Parameters::getInstance().checkConstraintGreaterZero(fac);
#endif
#endif
					//Self loop
					if (weight != 1) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
						parametric::Parameters::getInstance().checkConstraintLessOne(weight);
#endif
#endif
						fac /= weight;
					}

					//...and for each node newTarget ...
					//(we only have to look at nodes left from currNode, because all nodes
					//to the right of it have already been removed)
					for (int newTarget = 0; newTarget < currNodeCol; newTarget++) {
						//...increase the probability of going from srcNode to newTarget
						//by the probability prob(srcNode, currNode) * prob (currNode, newTarget) / (1 - prob of staying in currNode)
						if (prob[currNode][newTarget] != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
							parametric::Parameters::getInstance().checkConstraintGreaterZero(prob[currNode][newTarget]);
#endif
#endif
							prob[srcNode][newTarget] += fac * prob[currNode][newTarget];
						}
					}
				}
			}
		}
		//TODO make prob smaller as intermediate nodes are left out now

		if (DEFAULT_LOG_LEVEL <= TRACE_LOG_LEVEL) {
			stringstream ss;
			ss << "matrix after elimination of all intermediate nodes\n";
			//Column header
			ss << "         ";
			for (unsigned int j2 = 0; j2 < numIV + numOV; j2++) {
				stringstream tmp;
				tmp << "p" << vertices[j2]->oindex;
				ss << setw(6) << left << tmp.str();
				ss << " | ";
			}
			ss << "\n";

			//Probability table
			for (unsigned int j1 = 0; j1 < numIV; j1++) {
				stringstream tmp;
				tmp << "p" << vertices[j1 + numOV]->oindex << " = ";
				ss << setw(9) << left << tmp.str();
				PTYPE sum = 0;
				for (unsigned int j2 = 0; j2 < numIV + numOV; j2++) {
					ss << setw(6) << left << prob[j1][j2] << " | ";
					sum += prob[j1][j2];
				}
				ss << "\n";
				assert(sum == 1);
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
		}

		//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		//TODO this part of computing the probability on all copies takes long
		//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

		//Now only the input and output nodes are left in the matrix
		PTYPE probTmp[numIV][numOV + numIV];
		PTYPE probResult[numIV][numOV + numIV];
		//Initialize
		for (unsigned int j1 = 0; j1 < numIV; j1++) {
			for (unsigned int j2 = 0; j2 < numOV + numIV; j2++) {
				probResult[j1][j2] = 0;
			}
		}

		//We perform model checking for each input node independently
		//For each input node currNode...
		for (int currNode = numIV - 1; currNode >= 0; currNode--) {
			const unsigned int currNodeCol = currNode + numOV;

			//Initialize copied probability matrix
			for (unsigned int j1 = 0; j1 < numIV; j1++) {
				for (unsigned int j2 = 0; j2 < numOV + numIV; j2++) {
					probTmp[j1][j2] = prob[j1][j2];
				}
			}

			//We eliminate all other input nodes
			for (int middleNode = numIV - 1; middleNode >= 0; middleNode--) {
				if (middleNode == currNode) {
					continue; //Do not eliminate current input node
				}
				const unsigned int middleNodeCol = middleNode + numOV;

				//TODO only iterate over predecessors? Pro: no zero entries, contra: possible more edges which are not part of the scc
				for (unsigned int srcNode = 0; srcNode < numIV; srcNode++) {
					if ((int) srcNode == middleNode) {
						continue; //Can't eliminate self-loops, of course!
					}

					PTYPE weight = 1 - probTmp[middleNode][middleNodeCol];

					//...get the probability of taking a transition from node srcNode to middleNode...
					PTYPE fac = probTmp[srcNode][middleNodeCol];
					if (fac != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
						parametric::Parameters::getInstance().checkConstraintGreaterZero(fac);
#endif
#endif

						//Self loop
						if (weight != 1) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
							parametric::Parameters::getInstance().checkConstraintLessOne(weight);
#endif
#endif
							fac /= weight;
						}

						//...and for each node newTarget ...
						//(we only have to look at nodes left from middleNode, because all nodes
						//to the right of it have already been removed)
						for (unsigned int newTarget = 0; newTarget < middleNodeCol; newTarget++) {
							//...increase the probability of going from srcNode to newTarget
							//by the probability prob(srcNode, middleNode) * prob (middleNode, newTarget) / weight
							if (probTmp[middleNode][newTarget] != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
								parametric::Parameters::getInstance().checkConstraintGreaterZero(probTmp[middleNode][newTarget]);
#endif
#endif
								probTmp[srcNode][newTarget] += fac * probTmp[middleNode][newTarget];
								//LOG4CPLUS_TRACE(mclogger, "New probability: " << vertices[srcNode + numOV]->oindex << "->(" << vertices[middleNode + numOV]->oindex << ")->" << vertices[newTarget]->oindex << ": " << fac * probTmp[middleNode][newTarget] << ", All: " << probTmp[srcNode][newTarget]);

							}
						}

						if (middleNodeCol < currNodeCol) {
							probTmp[srcNode][currNodeCol] += fac * probTmp[middleNode][currNodeCol];
							//LOG4CPLUS_TRACE(mclogger, "New probability: " << vertices[srcNode + numOV]->oindex << "->(" << vertices[middleNode + numOV]->oindex << ")->" << vertices[currNodeCol]->oindex << ": " << fac * probTmp[middleNode][currNodeCol] << ", All: " << probTmp[srcNode][currNodeCol]);

						}
					}
				}
			}

			//Eliminate self loop for current node by scaling with sum of all outgoing transitions
			PTYPE sum = 1 - probTmp[currNode][currNodeCol];

#ifdef DEBUGGING
			sum = 0;
			for (unsigned int targetNode = 0; targetNode < numOV; targetNode++) {
				if (probTmp[currNode][targetNode] != 0) {
#ifdef USE_PARAMETRIC
					parametric::Parameters::getInstance().checkConstraintGreaterZero(probTmp[currNode][targetNode]);
#endif
					sum += probTmp[currNode][targetNode];
				}
			}
			assert(sum == 1 - probTmp[currNode][currNodeCol]);
#endif

			for (unsigned int targetNode = 0; targetNode < numOV; targetNode++) {
				if (probTmp[currNode][targetNode] != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
					parametric::Parameters::getInstance().checkConstraintGreaterZero(probTmp[currNode][targetNode]);
#endif
#endif
					//LOG4CPLUS_TRACE(mclogger,"Scaled: " << vertices[currNode + numOV]->oindex << "->" << vertices[targetNode]->oindex << ": " << probTmp[currNode][targetNode] / sum);
					probResult[currNode][targetNode] = probTmp[currNode][targetNode] / sum;
				}
			}
		}

		//Computation complete --> print final matrix if in tracing mode
		if (DEFAULT_LOG_LEVEL <= TRACE_LOG_LEVEL) {
			stringstream ss;
			ss << "final matrix:\n";

			//Probability table
			for (unsigned int j1 = 0; j1 < numIV; j1++) {
				stringstream tmp;
				tmp << "p" << vertices[j1 + numOV]->oindex << " = ";
				ss << setw(9) << left << tmp.str();
				PTYPE sum = 0;
				for (unsigned int j2 = 0; j2 < numIV + numOV; j2++) {
					ss << setw(6) << left << prob[j1][j2] << " | ";
					sum += prob[j1][j2];
				}
				ss << "\n";
				assert(sum == 1);
			}

			for (unsigned int j1 = 0; j1 < numIV; j1++) {
				stringstream tmp;
				tmp << "p" << vertices[j1 + numOV]->oindex << " = ";
				ss << setw(6) << left << tmp.str();
				bool first = true;
				for (unsigned int j2 = 0; j2 < numOV + numIV; j2++) {
					if (probResult[j1][j2] != 0) {
#ifdef DEBUGGING
#ifdef USE_PARAMETRIC
						parametric::Parameters::getInstance().checkConstraintGreaterZero(probResult[j1][j2]);
#endif
#endif
						if (first)
							first = false;
						else
							ss << " + ";
						stringstream tmp;
						tmp << probResult[j1][j2] << " * p" << vertices[j2]->oindex;
						ss << setw(10) << left << tmp.str();
					}
				}
				ss << endl;
			}
			LOG4CPLUS_TRACE(mclogger, ss.str());

			for (unsigned int j1 = 0; j1 < numIV; j1++) {
				for (unsigned int j2 = numOV; j2 < numOV + numIV; j2++) {
					assert(probResult[j1][j2] == 0.0);
				}
				PTYPE sum = 0.0;
				for (unsigned int j2 = 0; j2 < numOV; j2++) {
					sum += probResult[j1][j2];
				}

#ifdef USE_DOUBLE
				assert(sum>=0.99 && sum <= 1.01);
#else
				assert(sum == 1);
#endif
			}
		}

		//Add new abstract transitions from input to output nodes
		for (unsigned int i = 0; i < numOV; i++) {
			for (unsigned int j = 0; j < g->inputVertices.size(); j++) {
				PTYPE p = probResult[g->inputVertices[j]->dfs - numOV][i];

#if defined(USE_PARAMETRIC) && defined(USE_ABSTRACT_TRANSITIONS)
				p = parametric::Parameters::getInstance().addAbstractTransition(p);
#endif

				if (p != 0) {
#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
					parametric::Parameters::getInstance().checkConstraintGreaterZero(p);
#endif
					new Edge<PTYPE>(g->inputVertices[j], g->outputVertices[i], p, ABSTRACT);
				}
			}
		}

		g->state = ABSTRACT;
	}

	//JK: Maybe a bit too much DEBUGGING here... Might remove later
	LOG4CPLUS_TRACE(mclogger, "--leaving level of abstraction--");
}

//Recursively find SCCs in Graph g, using Tarjan's algorithm
void SccMc::strongComponents(Graph<PTYPE>* g) const {
	assert(g->inputVertices.size() > 0);

	//Initialize vertex state for Tarjan's algorithm
	for (unsigned int i = 0; i < g->allVertices.size(); i++) {
		Vertex<PTYPE>* v = g->allVertices[i];
		if (!v->isInputVertex) {
			v->dfs = -1;
			v->lowlink = -1;
			v->seen = true;
		}
	}

	vector<Vertex<PTYPE>*> S; //Stack of visited nodes
	int max_dfs = 0; //To keep track of the sequence in which nodes are detected

	//Start Tarjan's algo for all undiscovered nodes
	for (unsigned int i = 0; i < g->inputVertices.size(); i++) {
		Vertex<PTYPE>* s = g->inputVertices[i];
		for (unsigned int j = 0; j < s->outEdges.size(); j++) {
			Vertex<PTYPE>* v = s->outEdges[j]->target;
			//If seen is false, v has already been discovered in a previous
			//call of tarjan(..)
			if (v->seen) {
				assert(!(v->isInputVertex));
				assert(S.size() == 0);
				tarjan(v, g, S, max_dfs);
			}
		}
	}

	if (DEFAULT_LOG_LEVEL == TRACE_LOG_LEVEL) {
		for (unsigned int i = 0; i < g->allVertices.size(); i++)
			assert(!g->allVertices[i]->seen);
	}

	//tarjan adds the SCCs as subgraphs to g, so that we can now recursively
	//look for SCCs in the SCCs of g
	for (unsigned int i = 0; i < g->subgraphs.size(); i++)
		strongComponents(g->subgraphs[i]);
}

void SccMc::tarjan(Vertex<PTYPE> *v, Graph<PTYPE>* g, vtx_vector & S, int & max_dfs) const {
	if (!(v->g == g)) {
		LOG4CPLUS_ERROR(mclogger, "Wrong graph pointer set for " << (*v));
		// << ", corresponding graphs have ids " << g->graph_id << " and" << v->g->graph_id);
	}

	assert(v->g == g);

	v->dfs = max_dfs;
	v->lowlink = max_dfs;
	max_dfs++;
	S.push_back(v);
	v->seen = false;

	//DFS + computing lowlink values
	for (unsigned int i = 0; i < v->outEdges.size(); i++) {
		Vertex<PTYPE>* w = v->outEdges[i]->target;

		if (w->g == g && !w->isInputVertex) {
			if (w->seen) {
				tarjan(w, g, S, max_dfs);
				int low1 = v->lowlink;
				int low2 = w->lowlink;
				v->lowlink = (low1 < low2 ? low1 : low2);
			} else if (w->dfs != -1) {
				int low1 = v->lowlink;
				int dfs2 = w->dfs;
				v->lowlink = (low1 < dfs2 ? low1 : dfs2);
			}
		}
	}

	int low = v->lowlink;
	int dfs = v->dfs;
	//Have we found the root of an SCC?
	if (low == dfs) {
		if (S[S.size() - 1] == v) {
			assert(v->g == g);
			assert(!v->isInputVertex);
			v->dfs = -1;
			S.pop_back();
		} else {
			//We've found a non-trivial SCC, so we add it as subgraph
			Graph<PTYPE>& g2 = *(g->addSubgraph());
			int idx = S.size() - 1;
			while (true) {
				assert(idx >= 0);
				Vertex<PTYPE>* w = S[idx--];
				w->g = &g2;
				if (v == w)
					break;
			}

			//Iterate through all vertices of the SCC again to...
			for (unsigned int i = idx + 1; i < S.size(); i++) {
				//(1) add them to the subgraphs vertex list
				g2.allVertices.push_back(S[i]);

				//(2) find output vertices (vertices reachable from g2 in one step)
				for (unsigned int j = 0; j < S[i]->outEdges.size(); j++) {
					Vertex<PTYPE>* target = S[i]->outEdges[j]->target;
					if (target->g != &g2) {
						//Make sure we don't add an output vertex twice
						bool found = false;
						for (unsigned int k = 0; k < g2.outputVertices.size(); k++) {
							if (g2.outputVertices[k] == target) {
								found = true;
								break;
							}
						}
						if (!found) {
							LOG4CPLUS_TRACE(mclogger, "Adding the target of " << (* S[i]->outEdges[j]) << " to targets");
							g2.outputVertices.push_back(target);
						}
					}
				}

				//(3) find input vertices
				for (unsigned int j = 0; j < S[i]->inEdges.size(); j++) {
					assert(!S[i]->isInputVertex);
					if (S[i]->inEdges[j]->source->g != &g2) {
						g2.inputVertices.push_back(S[i]);
						S[i]->isInputVertex = true;
						S[i]->isConcretized = false;
						break;
					}
				}
			}

			S.resize(idx + 1);
		}
	}
}

void SccMc::reduceGraph(Graph<PTYPE>* g) const {
	assert(g->inputVertices.size() == 1 || g->outputVertices.size() == 1);

	if (g->outputVertices.size() == 1) {
		//Only one output vertex --> the probability to reach the output vertex
		//of the subgraph is 1.0 for all input vertices
		LOG4CPLUS_TRACE(mclogger, "reducegraph " << g->graph_id << ": single input/single output");

		for (unsigned int i = 0; i < g->inputVertices.size(); i++) {
#if defined(USE_PARAMETRIC) && defined(USE_ABSTRACT_TRANSITIONS)
			PTYPE prob = parametric::Parameters::getInstance().addAbstractTransitionOne();
#else
			PTYPE prob = 1.0;
#endif
			Edge<PTYPE>* e = new Edge<PTYPE>(g->inputVertices[i], g->outputVertices[0], prob);
			e->state = ABSTRACT;
			LOG4CPLUS_TRACE(mclogger, "added abstract edge " << (*e));
		}
		g->state = ABSTRACT;
	} else {
		//Only one input vertex --> we need one abstract edge from the input to
		//each output vertex --> need to compute weights for the abstract edges

		LOG4CPLUS_TRACE(mclogger, "reducegraph " << g->graph_id << ": single input/multiple output");
		LOG4CPLUS_TRACE(mclogger, "computing weights:");
		LOG4CPLUS_TRACE(mclogger, "Graph:" << (*g));

		g->inputVertices[0]->weight = 1.0;
		//Compute weight for each output vertex
		PTYPE sum = 0.0;
		for (unsigned int j = 0; j < g->outputVertices.size(); j++) {
			LOG4CPLUS_TRACE(mclogger, "Outer loop for " << (*g->outputVertices[j]));
			computeWeight(g->inputVertices[0], g->outputVertices[j]);
			sum += g->outputVertices[j]->weight;
		}
#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
		parametric::Parameters::getInstance().checkConstraintGreaterZero(sum);
#endif

		//Create the abstract edges
		//The actual probability has to be set to weight/sum to guarantee a total
		//probability of 1.0
		for (unsigned int j = 0; j < g->outputVertices.size(); j++) {
			//JK, Nov03/10: this assertion does not always hold!
			//assert(g.outputVertices[j]->weight > 0.0);
#if defined(USE_PARAMETRIC) && defined(USE_ABSTRACT_TRANSITIONS)
			PTYPE tmp(g->outputVertices[j]->weight / sum);
			PTYPE prob = parametric::Parameters::getInstance().addAbstractTransition(tmp);
#else
			PTYPE prob = g->outputVertices[j]->weight / sum;
#endif
			Edge<PTYPE>* e = new Edge<PTYPE>(g->inputVertices[0], g->outputVertices[j], prob);
			e->state = ABSTRACT;
			LOG4CPLUS_TRACE(mclogger, "added abstract edge " << (*e));
		}
		g->state = ABSTRACT;

		//clean up weights
		for (unsigned int i = 0; i < g->outputVertices.size(); i++)
			g->outputVertices[i]->weight = -1.0;

		for (unsigned int i = 0; i < g->allVertices.size(); i++)
			g->allVertices[i]->weight = -1.0;
	}
}
}
