/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <queue>

#include "State_Elim.h"
#include "elimorder/IndexOrder.h"
#include "elimorder/DistanceOrder.h"
#include "elimorder/DistanceInitialOrder.h"
#include "elimorder/RandomOrder.h"
#include "elimorder/ConnectivityOrder.h"

#ifdef USE_PARAMETRIC
#include "../data/Parameters.h"
#endif

namespace scc_cex {

void StateElim::modelCheck_common(Graph<PTYPE> *g) {
	elimOrder->init(g);

	//Iteratively eliminate states
	Vertex<PTYPE>* v = NULL;
	while (elimOrder->getNextVertex(g, v)) {
		updateProb(g, v);
	}
	g->state = ABSTRACT;

#ifdef DEBUGGING
	unsigned int size = 0;
	for (vtx_it iter = g->allVertices.begin(); iter != g->allVertices.end(); iter++) {
		if (!(*iter)->eliminated) {
			size++;
		}
	}

	assert(size == g->inputVertices.size());
#endif

	//Delete self loop for input vertex
	Vertex<PTYPE> *input = g->inputVertices.front();
	PTYPE selfLoop = 1 - input->getSelfLoopProb();

#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
	parametric::Parameters::getInstance().checkConstraintGreaterZero(selfLoop);
#endif

	for (unsigned int i = 0; i < input->outEdges.size(); i++) {
		input->outEdges[i]->prob /= selfLoop;
	}

	LOG4CPLUS_INFO(mclogger, "MODEL CHECKING RESULT:\n" << *g);
}

/**
 * Update probabilities for adjacent vertices of v
 * @param g graph
 * @param v vertex
 */
void StateElim::updateProb(Graph<PTYPE> *g, Vertex<PTYPE> *v) {
	LOG4CPLUS_DEBUG(mclogger, "Eliminating " << v->oindex);

	//Search for self loop
	PTYPE selfProb = v->getSelfLoopProb();
	if (selfProb != 0.0) {

#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
		parametric::Parameters::getInstance().checkConstraintGreaterZero(selfProb);
#endif

		//LOG4CPLUS_TRACE(mclogger, "Self loop for " << v->oindex << " with probability: " << selfProb);
	}
	selfProb = 1 - selfProb;

#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
	parametric::Parameters::getInstance().checkConstraintGreaterZero(selfProb);
#endif

	//Update edge probabilities for adjacent vertices
	for (edge_it iterIn = v->inEdges.begin(); iterIn != v->inEdges.end(); iterIn++) {
		Vertex<PTYPE> *in = (*iterIn)->source;
		if (in == v) {
			//No selfloop
			continue;
		}

		PTYPE probTmp = (*iterIn)->prob / selfProb;

		for (edge_it iterOut = v->outEdges.begin(); iterOut != v->outEdges.end(); iterOut++) {
			Vertex<PTYPE> *out = (*iterOut)->target;
			if (out == v) {
				//No selfloop
				continue;
			}

			//Update probability
			PTYPE prob = probTmp * (*iterOut)->prob;

#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
			parametric::Parameters::getInstance().checkConstraintGreaterZero(prob);
			if (prob != 1.0) {
				parametric::Parameters::getInstance().checkConstraintLessOne(prob);
				}
#endif

			//Search for existing edge
			//TODO make more efficient
			bool existingEdge = false;
			for (edge_it iterEx = in->outEdges.begin(); iterEx != in->outEdges.end(); iterEx++) {
				if ((*iterEx)->target == out) {
					//Edge already exists
					(*iterEx)->prob += prob;

#if defined(DEBUGGING) && defined(USE_PARAMETRIC)
					parametric::Parameters::getInstance().checkConstraintGreaterZero((*iterEx)->prob);
					if ((*iterEx)->prob != 1.0) {
						parametric::Parameters::getInstance().checkConstraintLessOne((*iterEx)->prob);
					}
#endif

					//(*iterEx)->state = ABSTRACT;
					existingEdge = true;
					//LOG4CPLUS_TRACE(mclogger, "Updated probability for " << in->oindex << " -> " << out->oindex << " with " << prob << ": " << (*iterEx)->prob);
					break;
				}
			}

			if (!existingEdge) {
				//New Edge
				if (in == out) {
					in->selfLoop = new Edge<PTYPE>(in, out, prob, ABSTRACT);
				} else {
					new Edge<PTYPE>(in, out, prob, ABSTRACT);
				}
				//LOG4CPLUS_TRACE(mclogger, "Added new probability for " << in->oindex << " -> " << out->oindex << ": " << prob);
			}
		}
	}

#ifdef DEBUGGING
	for (edge_it iterIn = v->inEdges.begin(); iterIn != v->inEdges.end(); iterIn++) {
		Vertex<PTYPE> *in = (*iterIn)->source;
		if (in == v) {
			//No selfloop
			continue;
		}
		PTYPE prob = 0.0;
		for (edge_it iter = in->outEdges.begin(); iter != in->outEdges.end(); iter++) {
			if ((*iter)->target != v) {
				prob += (*iter)->prob;
			}
		}
		assert(prob == 1.0);
	}
#endif

	//Eliminate vertex
	v->clear();
	v->eliminated = true;

}

void StateElim::setEliminationOrder(EliminationOrder::ORDER order, bool increasing) {
	switch (order) {
	case EliminationOrder::INDEX:
		elimOrder = new IndexOrder(increasing);
		break;
	case EliminationOrder::RANDOM:
		elimOrder = new RandomOrder(increasing);
		break;
	case EliminationOrder::DISTANCE:
		elimOrder = new DistanceOrder(increasing);
		break;
	case EliminationOrder::DISTANCE_INITIAL:
		elimOrder = new DistanceInitialOrder(increasing);
		break;
	case EliminationOrder::CONNECTIVITY:
		elimOrder = new ConnectivityOrder(increasing);
		break;
	default:
		//Should not come here
		assert(false);
		break;
	}
}
}
