/*
 * AlgorithmWrapper.cpp
 *
 *  Created on: 13.02.2012
 *      Author: Markus Bals
 */

#include "ignf.h"

namespace IGNF {

AlgorithmWrapper::AlgorithmWrapper(map<string,bool> options) {
	m_options = options;
}

void AlgorithmWrapper::runOnGrammarFile(string inputfilename) {
	GrammarReader reader;
	reader.readGrammarFile(inputfilename);

	while ( reader.good() ) {
		insertProduction(reader.getNextProduction());
	}

	if ( !m_options["console"] ) {
		cout << "Final "; showProductionsIncremental();
		showRuntimes();
	}
}

bool AlgorithmWrapper::append(string production) {
	GrammarReader reader;
	StringProduction strProd = reader.readProduction(production);
	if ( strProd.first.empty() || strProd.second.empty() )
		return false;
	insertProduction(strProd);
	return true;
}

void AlgorithmWrapper::insertProduction(StringProduction strProduction) {
	string plainStr = StringGrammar::joinStringProduction(strProduction);
	m_grammar.addProduction(strProduction);

	cout << "Inserting " << plainStr << endl;
	Runtimes times;

	m_clock.resetClock();
	m_simple.runOnGrammar(m_grammar);
	times.simple = m_clock.getTimeval();

	m_clock.resetClock();
	m_basic.runOnGrammar(m_grammar);
	times.basic = m_clock.getTimeval();

	m_clock.resetClock();
	m_incremental.DT38_extendGrammar( strProduction );
	times.incremental = m_clock.getTimeval();

	cout << endl;

	m_insertedProductions.push_back( plainStr );
	m_performance.push_back(times);
}

void AlgorithmWrapper::showEncodingTable() {
	m_incremental.showEncodingTable();
}

void AlgorithmWrapper::showProductionsSimple() {
	m_simple.showProductions();
}

void AlgorithmWrapper::showProductionsBasic() {
	m_basic.showProductions();
}

void AlgorithmWrapper::showProductionsIncremental() {
	m_incremental.showProductions();
}

void AlgorithmWrapper::showOriginalProductions() {
	cout << "Original Productions:\n";
	cout << m_grammar.getProductionsString(set<Production>(),true) << endl;
}

void AlgorithmWrapper::showRuntimes() {
	char buffer[100];
	timeval sumSimple;
	timeval sumBasic;
	timeval sumIncremenal;

	sumSimple.tv_sec = 0;
	sumSimple.tv_usec = 0;
	sumBasic.tv_sec = 0;
	sumBasic.tv_usec = 0;
	sumIncremenal.tv_sec = 0;
	sumIncremenal.tv_usec = 0;

	cout << "/-----------------------------------------------------\\\n";
	sprintf(buffer, "| %-12s | %-10s | %-10s | %-10s |\n",
			"production", "simple", "basic", "incr.");
	cout << buffer;
	cout << "|-----------------------------------------------------|\n";

	for(size_t i=0; i<m_performance.size(); i++) {

		sprintf(buffer, "| %-12s | %10s | %10s | %10s |\n",
				m_insertedProductions[i].c_str(),
				Clock::time2string(m_performance[i].simple).c_str(),
				Clock::time2string(m_performance[i].basic).c_str(),
				Clock::time2string(m_performance[i].incremental).c_str()
		);
		cout << buffer;
	}
	cout << "|-----------------------------------------------------|\n";
	sprintf(buffer, "| %-12s | %-10s | %-10s | %-10s |\n",
			"production", "simple sum", "basic sum", "incr. sum");
	cout << buffer;

	cout << "|-----------------------------------------------------|\n";
	for(size_t i=0; i<m_performance.size(); i++) {

		sumSimple = Clock::timesum(sumSimple, m_performance[i].simple);
		sumBasic = Clock::timesum(sumBasic, m_performance[i].basic);
		sumIncremenal = Clock::timesum(sumIncremenal, m_performance[i].incremental);

		sprintf(buffer, "| %-12s | %10s | %10s | %10s |\n",
				m_insertedProductions[i].c_str(),
				Clock::time2string(sumSimple).c_str(),
				Clock::time2string(sumBasic).c_str(),
				Clock::time2string(sumIncremenal).c_str()
		);
		cout << buffer;
	}

	cout << "\\-----------------------------------------------------/\n";
}

} /* namespace IGNF */
