/*
 * BasicAlgorithm.cpp
 *
 *  Created on: 23.01.2012
 *      Author: Markus Bals
 */

#include "ignf.h"

namespace IGNF {

BasicAlgorithm::BasicAlgorithm() {
	clearAlgorithm();
}

BasicAlgorithm::~BasicAlgorithm() {
}

void BasicAlgorithm::runOnGrammarFile(string filename) {
	m_grammar.initFromFile(filename);
    runOnGrammar(m_grammar);
}


void BasicAlgorithm::runOnGrammar(StringGrammar grammar) {
	cout << "Starting basic algorithm.\n";
	clearAlgorithm();
	m_grammar = grammar;

	if ( m_grammar.getProductions().size() < 1 ) {
#ifdef DEBUG
		cout << "Grammar is empty, so there is nothing to do.\n";
#endif
		return;
	}
#ifdef DEBUG
	cout << m_grammar.getEncodingTableString();
	cout << m_grammar.getProductionsString();
	cout << "\nGrammar was read, starting algorithm by initializing worklists\n";
#endif

	initWorklists();
#ifdef DEBUG
	cout << "\nProcessing ordering list:\n";
#endif
	processWorklist(&m_orderingList);
#ifdef DEBUG
	cout << "\nProcessing greibach list:\n";
#endif
	processWorklist(&m_greibachList);
#ifdef DEBUG
	cout << "\nProcessing final list:\n";
#endif
	processWorklist(&m_finalList);
	cout << "Basic algorithm terminated successfully.\n";
}

void BasicAlgorithm::showProductions() {
	cout << "Basic Algorithm Productions:\n"
			<< m_grammar.getProductionsString(set<Production>(), true);
}

void BasicAlgorithm::clearAlgorithm() {
	m_orderingList = Worklist<WorklistProduction>(false);
	m_greibachList = Worklist<WorklistProduction>(true);
	m_finalList    = Worklist<WorklistProduction>(false);
}

void BasicAlgorithm::initWorklists() {
	checkForWorklist(m_grammar.getProductions());
}


void BasicAlgorithm::checkForWorklist(Production production) {
#ifdef DEBUG
	cout << "  checking WL for (" << m_grammar.getProductionString(production) << "): ";
#endif
	if ( !StringGrammar::ordered(production) || StringGrammar::recursive(production) ) {
#ifdef DEBUG
		cout << "put on ordering list\n";
#endif
		m_orderingList.insertItem(WorklistProduction(production));
	}
	else if ( !StringGrammar::greibach(production)) {
		if ( IS_RECURSIVE_SYMBOL(production.first) ) {
#ifdef DEBUG
			cout << "put on final list\n";
#endif
			m_finalList.insertItem(WorklistProduction(production));
		}
		else {
#ifdef DEBUG
			cout << "put on greibach list\n";
#endif
			m_greibachList.insertItem(WorklistProduction(production));
		}
	}
#ifdef DEBUG
	else {
		cout << "already in GNF, nothing to do\n";
	}
#endif
}

void BasicAlgorithm::checkForWorklist(set<Production> productions) {
	set<Production>::iterator it = productions.begin();

	while ( it != productions.end() ) {
		checkForWorklist(*it);
		it++;
	}
}

void BasicAlgorithm::processWorklist(Worklist<WorklistProduction> *worklist) {
	while ( !worklist->empty() ) {
		WorklistProduction wlItem = worklist->popItem();
		Production production = wlItem.getProduction();
#ifdef DEBUG
		cout << "* processing production " << m_grammar.getProductionString(production) << endl;
#endif
		if ( StringGrammar::recursive(production) )
			removeDirectLeftRecursion(production);
		else
			replaceByLeftDerivations(production);
	}
}


void BasicAlgorithm::replaceByLeftDerivations(Production production) {
	set<Production> inserted = m_grammar.getProductions(production.second[0]);
	set<Production> newProductions =
			StringGrammar::getLeftDerivations(production, inserted);
#ifdef DEBUG
	cout << "removing " << m_grammar.getProductionString(production) <<
			" and inserting " << m_grammar.getProductionsString(newProductions);
#endif
	m_grammar.removeProduction(production);
	m_grammar.addProductions(newProductions);
	checkForWorklist(newProductions);
}

void BasicAlgorithm::removeDirectLeftRecursion(Production production) {
	set<Production> nonRecursive = m_grammar.getProductions(production.first);
	set<Production>::iterator it = nonRecursive.begin();

	while ( it != nonRecursive.end() ) {
		if ( StringGrammar::recursive(*it) )
			nonRecursive.erase(*it);
		it++;
	}

	bool rrExists = m_grammar.checkRecursiveSymbol(production.first);

	set<Production> newProductions =
			StringGrammar::getRightRecursiveReplacement(production, nonRecursive, rrExists);
#ifdef DEBUG
	cout << "  removing " << m_grammar.getProductionString(production) <<
			" and inserting " << m_grammar.getProductionsString(newProductions);
#endif
	m_grammar.removeProduction(production);
	m_grammar.addProductions(newProductions);
	checkForWorklist(newProductions);
}

} /* namespace IGNF */
