//============================================================================
// Name        : IGNF.cpp
// Author      : Markus Bals
// Description : Incremental Greibach Normal Form
//               Console application to run both the basic and incremental
//               version of the algorithm constructing the GNF
//============================================================================

#include <stdio.h>
#include <stdlib.h>

#include "ignf.h"

map<string, bool> algOptions;		//store command line options here

void printHelp();
bool processInput(string, IGNF::AlgorithmWrapper&);
bool checkOptions(string);
bool checkFile(string);

int main(int argc, char* argv[]) {
	algOptions.clear();
	algOptions["console"] = false;

	//need at least an option or a file name
	if ( argc < 2 ) {
		printHelp();
		return EXIT_SUCCESS;
	}


	//show help if requested
	string arg = argv[1];
	if ( arg == "help" || arg.find("--") == 0 ) {
		printHelp();
		return EXIT_SUCCESS;
	}

	//check the command line parameters
	string filename = "";
	for(int i=1; i<argc; i++) {
		if ( !checkOptions(argv[i]) )
			filename = argv[i];
	}

	//check if there was a valid file name supplied
	bool validFile = (filename.length() > 0 && checkFile(filename));

	//in file mode we require a valid file
	if ( !validFile && !algOptions["console"] ) {
		cout << "In file mode a valid grammar file has to be supplied.\nExiting.\n";
		return EXIT_FAILURE;
	}

	IGNF::AlgorithmWrapper wrapper(algOptions);

	if ( validFile )
		wrapper.runOnGrammarFile(filename);

	if ( !algOptions["console"] )
		return EXIT_SUCCESS;

	cout << "Console mode: q=quit, o=original productions, r=runtimes, e=encoding\n"
		 << "              s=simple result, b=basic result, i=incremental result\n"
		 << "ignf> ";
	char inputBuffer[256];
	cin.getline(inputBuffer, 256);

	while ( processInput(inputBuffer, wrapper) ) {
		cout << "ignf> ";
		cin.getline(inputBuffer, 256);
	}
	return EXIT_SUCCESS;
}


void printHelp() {
	cout << "Usage: ignf [OPTIONS] [FILE]\n"
			<< "Executes the Greibach algorithms on the grammar defined in the\n"
			<< "specified file.\n"
			<< "Valid options are:\n"
			<< "  -f\t file mode - executes the basic and incremental algorithm\n"
			<< "    \t parallel on supplied file and shows statistics\n"
			<< "  -c\t console mode - allows to insert additional productions and\n"
			<< "    \t query information; in console mode enter ? for further help\n"
			<< endl;
}


bool processInput(string inputStr, IGNF::AlgorithmWrapper& algWrapper) {
	if ( inputStr.length() < 1 || inputStr == "h" || inputStr == "?") {
		cout << "Enter a valid production that will be added or one of the following commands:\n"
			 << "  q = quit\n  e = show encoding table\n"
			 << "  r = show runtime table\n  o = show original productions\n"
			 << "  s = show result of the basic algorithm (simple version)\n"
			 << "  b = show result of the basic algorithm (worklist improved)\n"
			 << "  i = show result of the incremental algorithm\n";
		return true;
	}

	if ( inputStr == "q")
		return false;

	if ( inputStr == "e" ) {
		algWrapper.showEncodingTable();
		return true;
	}

	if ( inputStr == "p" || inputStr == "i" ) {
		algWrapper.showProductionsIncremental();
		return true;
	}

	if ( inputStr == "b" ) {
		algWrapper.showProductionsBasic();
		return true;
	}

	if ( inputStr == "s" ) {
		algWrapper.showProductionsSimple();
		return true;
	}

	if ( inputStr == "r" ) {
		algWrapper.showRuntimes();
		return true;
	}

	if ( inputStr == "o" ) {
		algWrapper.showOriginalProductions();
		return true;
	}

	//no command, so assume user entered a new production. try to append
	if ( !algWrapper.append(inputStr) )
		cout << "Not a valid production.\n";

	return true;
}

bool checkOptions(string arg) {
	if ( arg[0] != '-' )
		return false;
	for(size_t i=1; i<arg.length(); i++) {
		switch ( arg[i] ) {
		case 'c': algOptions["console"] = true; break;
		case 'f': algOptions["console"] = false; break;
		default: cout << "unknown option '" << arg[i] << "' is ignored\n"; break;
		}
	}

	return true;
}

bool checkFile(string filename) {
	ifstream ifs;
	ifs.open(filename.c_str(), ifstream::in);

	if ( !ifs.is_open() ) {
		cout << "Cannot open file \"" << filename << "\"\n";
		return false;
	}
	ifs.close();
	return true;
}
