/*
 * IncrementalAlgorithm.cpp
 *
 *  Created on: 06.02.2012
 *      Author: Markus Bals
 */

#include "ignf.h"

namespace IGNF {

IncrementalAlgorithm::IncrementalAlgorithm() {
	m_orderingList = Worklist<TreeNode*>(false);
	m_greibachList = Worklist<TreeNode*>(true);
	m_finalList    = Worklist<TreeNode*>(false);
}

IncrementalAlgorithm::~IncrementalAlgorithm() {
}

void IncrementalAlgorithm::runOnGrammarFile(string filename) {
	GrammarReader reader;
	reader.readGrammarFile(filename);

	while ( reader.good() ) {
		DT38_extendGrammar( reader.getNextProduction() );
	}
}

void IncrementalAlgorithm::DT38_extendGrammar(string raw) {
	GrammarReader reader;
	DT38_extendGrammar(reader.readProduction(raw));
}

/*
 * The main entry point of the incremental algorithm.
 */
void IncrementalAlgorithm::DT38_extendGrammar(StringProduction production) {

	cout << "Starting incremental algorithm.\n";

	TreeNode* newNode = DT32_insertProduction(production);

	//follow forward pointers of the tree holding the new production
	DT37_followPointers(m_forest[newNode->getLHS()], UPforward);
	m_forest.resetFollowedMarkers();
	DT36_processWorklist(m_orderingList, orderingPass);

	//follow backward pointers from all modified trees
	set<Tree*> trees = m_forest.getTrees();
	set<Tree*>::iterator it = trees.begin();
	while ( it != trees.end() ) {
		if ( (*it)->hasNewNodes() )
		  DT37_followPointers(*it, UPbackward);
		it++;
	}
	m_forest.resetFollowedMarkers();
	DT36_processWorklist(m_greibachList, greibachPass);

	//follow final pointers from all modified trees
	it = trees.begin();
	while ( it != trees.end() ) {
		if ( (*it)->hasNewNodes() )
		  DT37_followPointers(*it, UPfinal);
		it++;
	}
	m_forest.resetFollowedMarkers();
	DT36_processWorklist(m_finalList, finalPass);

	cleanup();
	cout << "Incremental algorithm terminated successfully.\n";
}

TreeNode* IncrementalAlgorithm::DT32_insertProduction(Production encoded) {
	TreeNode* newNode = m_forest.insertProduction(encoded);
	insertedNewRootChild(newNode);
	return newNode;
}


TreeNode* IncrementalAlgorithm::DT32_insertProduction(StringProduction production) {
	TreeNode* newNode = m_forest.insertProduction(production);
	insertedNewRootChild(newNode);
	return newNode;
}


void IncrementalAlgorithm::DT33_checkPointerAndWorklist(TreeNode* node) {
	Production production = node->getProduction();

	//mark the tree as modified, since the procedure is called for new nodes only
	m_forest[production.first]->setNewNodes(true);
#ifdef DEBUG
	cout << "  checking PW: (" << m_forest.decodeNode2String(node) << ") ";
#endif
	if ( StringGrammar::greibach(production) ) {
#ifdef DEBUG
		cout << "already in GNF, nothing to do\n";
#endif
		return;
	}

	Symbol leftmost = node->getLeftmostRHS();

	if ( !StringGrammar::ordered(production) || StringGrammar::recursive(production) ) {
		m_orderingList.insertItem(node);
#ifdef DEBUG
		cout << "to ordering list, ";
#endif
		if ( !StringGrammar::recursive(production) ) {
			m_forest[leftmost]->addForwardPointer(node);
#ifdef DEBUG
		    cout << "added fwd ptr from " << leftmost << endl;
#endif
		}
#ifdef DEBUG
		else {
			cout << "no pointer (recursive)\n";
		}
#endif
	}
	else if ( !StringGrammar::greibach(production) ) {
		if ( IS_RECURSIVE_SYMBOL(production.first) ) {
			m_finalList.insertItem(node);
			m_forest[leftmost]->addFinalPointer(node);
#ifdef DEBUG
			cout << "to final list, added fin ptr from " << leftmost << "\n";
#endif
		}
		else {
			m_greibachList.insertItem(node);
			m_forest[leftmost]->addBackwardPointer(node);
#ifdef DEBUG
			cout << "to greibach list, added bck ptr from " << leftmost << "\n";
#endif
		}
	}
}

void IncrementalAlgorithm::DT36_processWorklist(Worklist<TreeNode*>& worklist, PassType pass) {
#ifdef DEBUG
	switch ( pass ) {
	case orderingPass: cout << "ORDERING PASS:\n"; break;
	case greibachPass: cout << "GREIBACH PASS:\n"; break;
	case finalPass: cout << "FINAL PASS:\n"; break;
	default: cout << "Warning: unknown pass!\n"; break;
	}
#endif
    while ( !worklist.empty() ) {
        TreeNode* item = worklist.popItem();
#ifdef DEBUG
        cout << "  processing " << m_forest.decodeNode2String(item) << "\n";
#endif
        if ( StringGrammar::recursive(item->getProduction()) )
        	DT35_removeRecursion(item);
        else
        	DT39_leftDerivation(item, pass);
    }
}

void IncrementalAlgorithm::DT37_followPointers(Tree* tree, UsgPtrType ptrType) {
	if ( tree->wasFollowed() )
		return;
	tree->setFollowed(true);
	set<TreeNode*> destinations = tree->getUsagePointerDestinations(ptrType);
	set<TreeNode*>::iterator it = destinations.begin();
	while( it != destinations.end() ) {
		TreeNode* node = *it;
		if ( !node->isNew() ) {
			if ( ptrType == UPforward )
				m_orderingList.insertItem(node);
			else if ( ptrType == UPbackward )
				m_greibachList.insertItem(node);
			else if ( ptrType == UPfinal )
				m_finalList.insertItem(node);
		}
		DT37_followPointers(m_forest[node->getLHS()], ptrType);
		it++;
	}
}

void IncrementalAlgorithm::DT39_leftDerivation(TreeNode* node, PassType pass) {
#ifdef DEBUG
	cout << "  left derivation applied to ";
#endif
	set<Production> replacements, derivatives;
	Symbol leftmostRHS = node->getLeftmostRHS();

	//determine the productions that are used for replacement
	//(c.f. Algorithm 3.9 on page 29 in the Diploma Thesis)
	if ( node->isNew() && pass == orderingPass ) {
#ifdef DEBUG
		cout << "new node in ordering pass\n";
#endif
		replacements = m_forest[leftmostRHS]->getFONProductions();
	}
	else {
		replacements = m_forest[leftmostRHS]->getRepresentedProductions();
		if ( !node->isNew() ) {
#ifdef DEBUG
			cout << "old node in pass " << pass << endl;
#endif
			set<Production>::iterator it = replacements.begin();
			while ( it != replacements.end() ) {
				if ( m_oldProductions[leftmostRHS].find(*it) != m_oldProductions[leftmostRHS].end() )
					replacements.erase(it);
				it++;
			}
		}
#ifdef DEBUG
		else {
		  cout << "new node in greibach or final pass\n";
		}
#endif
	}

	//now do the actual replacement
	replaceByLeftDerivations(node, replacements);
}

void IncrementalAlgorithm::DT35_removeRecursion(TreeNode *node) {
#ifdef DEBUG
	cout << "  removing recursion\n  ";
#endif
	node->setBlind();
	Production production = node->getProduction();

	if ( production.second.size() < 2 ) {
#ifdef DEBUG
		cout << "Warning: encountered useless production " << m_forest.decodeNode2String(node) << "\n";
#endif
		return;
	}

	//introduce the new right recursive symbol with corresponding tree and new node
	createProductionsOfRRSymbol(production);

	//right recursive copies are handled by marking the tree as extended
	m_forest[production.first]->extend();

}

void IncrementalAlgorithm::showEncodingTable() {
	cout << m_forest.getEncodingTableString();
}

void IncrementalAlgorithm::showProductions() {
	cout << "Incremental Algorithm Productions:\n" << getCurrentProductionsString() << "\n";
}


void IncrementalAlgorithm::cleanup() {
#ifdef DEBUG
	cout << "cleanup\n";
#endif
	set<TreeNode*>::iterator it = m_newNodes.begin();
	while ( it != m_newNodes.end() ) {
		(*it)->setOld();
		it++;
	}
	m_newNodes.clear();
	m_forest.resetNewMarkers();
	memorizeOldProductions();
}

void IncrementalAlgorithm::memorizeOldProductions() {
	set<Symbol> labels = m_forest.getRootLabels();
	set<Symbol>::iterator it = labels.begin();
	while ( it != labels.end() ) {
		set<Production> curProductions = m_forest[*it]->getRepresentedProductions();
		m_oldProductions[*it].insert(curProductions.begin(), curProductions.end());
		it++;
	}
}


void IncrementalAlgorithm::insertedNewRootChild(TreeNode* newNode) {
	m_newNodes.insert(newNode);
#ifdef DEBUG
	cout << "Inserted new root child: " << m_forest.decodeNode2String(newNode) << "\n";
#endif
	DT33_checkPointerAndWorklist( newNode );
}


void IncrementalAlgorithm::replaceByLeftDerivations(TreeNode* node, set<Production> replace) {
#ifdef DEBUG
	cout << "  adding derivatives:\n";
#endif
	Production oldProduction = node->getProduction();
	set<Production>::iterator it = replace.begin();

	while ( it != replace.end() ) {
		RHS newRHS = (*it).second;
		newRHS.insert(newRHS.end(), oldProduction.second.begin()+1, oldProduction.second.end());

		TreeNode *newNode = node->addChild(Production(oldProduction.first, newRHS));
		m_newNodes.insert( newNode );
		DT33_checkPointerAndWorklist(newNode);
		it++;
	}
}

void IncrementalAlgorithm::createProductionsOfRRSymbol(Production production) {
	Symbol recursiveLHS = RECURSIVE_SYMBOL(production.first);

	//create only one node because the second production is created on query (extended tree)
	Production rrProduction;
	rrProduction.first = recursiveLHS;
	rrProduction.second.assign(production.second.begin()+1, production.second.end());

	//actually insert the new production
	//(creates a new node, adds it to "new Nodes", checks pointer and worklist)
	DT32_insertProduction(rrProduction);

	//trees of right recursive symbols are always extended
	m_forest[recursiveLHS]->extend();
}

string IncrementalAlgorithm::getCurrentProductionsString() {
	stringstream ss;
	map<Symbol, set<Production> >::iterator it = m_oldProductions.begin();
	while ( it != m_oldProductions.end() ) {
		ss << m_forest.getProductionsString((*it).second, true);
		it++;
	}
	return ss.str();
}


} /* namespace IGNF */
