grammar HRG;
/*
	this grammar contains the lexer and basic parser, it produces an abstract syntax tree that is used by all other grammars.
*/

options
{
	// produces AST
	output=AST;
	ASTLabelType=CommonTree;
	// k=2; // is sufficient, k=1 will break at "node: ID | ID ID"
}

tokens
{
	// additional tokens, that are not used by the lexer but produced by rewrite rules
	RULE;
	DEFINITION;
	EDGE;
	NODE;
	TYPE;
	NAME;
}

@parser::header
{
	package de.rwth.aachen.i2.graphreduction.newgrammar;
}
@lexer::header
{
	package de.rwth.aachen.i2.graphreduction.newgrammar;
}

@members
{
	/**
	 * try to recover from an error state.
	 * simply ignore tokens that do not occur in the follow set of the current rule.
	 * calls {@link #syncToSet(BitSet)} with the appropriate follow set.
	 */
	protected void syncToSet()
	{
		syncToSet(state.following[state._fsp]);
	}

	/**
	 * try to recover. ignore tokens until next token is contained in the given follow set.
	 *
	 * @param follow set of allowed inputs in follow set
	 */
	protected void syncToSet(BitSet follow) 
	{
		int mark = -1, tokenid = -1;
		Token token;
		try {
			mark = input.mark();
			// consume all input up to an element of the follow set
			while (! follow.member(input.LA(1)) ) {
				tokenid = input.LA(1);
				token = input.LT(1);
				if  (tokenid == Token.EOF) { // no chance to recover
					input.rewind();
					mark = -1;
					return;
				}
				input.consume();
				this.emitErrorMessage("Ignored token to recover: \"" + token.getText() + "\" (Position " + token.getLine() + ":" + token.getCharPositionInLine() + ")" );
			}
		} catch (Exception e) {
			// really screwed up, let the default mechanism recover
		} finally {
			// release the mark
			if  (mark != -1) input.release(mark);
		}
	}
}

rules
	: rule* EOF
		-> 	(^( RULE rule ))*
;

sync
@init { syncToSet(); }
	: 
;

rule
	: ID COLON tuple (RESULT_MARK ID tuple)*
		-> 	^(DEFINITION ^(TYPE ID) tuple)
			( ^(EDGE ^(TYPE ID) tuple ))* 
;

tuple
	: TUPEL_START node sync ( TUPEL_SEP node sync )* TUPEL_END
		-> 	(^( NODE node ))+ 
;

node
	: ID 
		-> 	^(NAME ID)
	| ID ID 
		-> 	^(TYPE ID) ^(NAME ID)
	| NULL
		-> NULL
;

COLON : ':';
TUPEL_START : '(';
TUPEL_SEP : ',';
TUPEL_END : ')';
RESULT_MARK : '->';
NULL : ('n'|'N') ('u'|'U') ('l'|'L') ('l'|'L') ;

ID : ('a'..'z' | 'A'..'Z' | '0'..'9')+ ;
WHITESPACE : (' '|'\r'|'\t'|'\n')+ { $channel = HIDDEN; } ;
COMMENT : ( '/*' .* '*/' | '//' .* ('\n'|'\r')+ ) { $channel = HIDDEN; } ;
REST : . ;