tree grammar HRGCollectorGrammar;

options {
	language=Java;
	tokenVocab=HRG;
	output=AST;
	ASTLabelType=CommonTree;
}

@header {
package de.rwth.aachen.i2.graphreduction.newgrammar;
	
import de.rwth.aachen.i2.graphreduction.antlr.HRGUnknownTypeException;
import de.rwth.aachen.i2.graphreduction.antlr.HRGUnknownSelectorException;
import de.rwth.aachen.i2.graphreduction.antlr.HRGRankException;
import de.rwth.aachen.i2.graphreduction.antlr.HRGNotTypedNonterminalException;
import de.rwth.aachen.i2.graphreduction.grammar.Alphabet;
import de.rwth.aachen.i2.graphreduction.grammar.HRGrammar;
import de.rwth.aachen.i2.graphreduction.hypergraph.HyperGraph;
import de.rwth.aachen.i2.graphreduction.grammar.NodeType;
import de.rwth.aachen.i2.graphreduction.grammar.Nonterminal;
import de.rwth.aachen.i2.graphreduction.grammar.ReductionTentacle;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.*;

import java.util.Map;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Arrays;
}

@members {
	static class NonterminalInfo{
		String label;
		NodeType[] typeSequence;
		boolean[] entrance;  
		Nonterminal nonterminal;
      
      public String toString()
      {
      	String res = "NonterminalInfo " + this.label + ":";
      	res += "; nonterminal: " + this.nonterminal;
      	res += "; typeSequence: ";
      	if (this.typeSequence == null) res += "null";
      	else for (int i=0; i<this.typeSequence.length; i++) res += this.typeSequence[i] + ", ";
      	res += "; entrance: ";
      	if (this.entrance == null) res += "null";
      	else for (int i=0; i<this.entrance.length; i++) res += this.entrance[i] + ", ";
      	return res;
      }
	}
	static class NodeInfo{
		String label;
		NodeType type;
		boolean reduct;
		int id;   
      
      public String toString()
      {
      	return "NodeInfo: label=" + this.label + ", type=" + this.type + ", reduct=" + this.reduct + ", id=" + this.id;
      }    
	}

	public Map<String, NonterminalInfo> nonterminalMap;
	
	private HRGrammar grammar; 
	private Alphabet alphabet;
	
	public HRGCollectorGrammar(TreeNodeStream input, HRGrammar grammar)
	{
		this(input);
		
		this.grammar = grammar;
		this.alphabet = this.grammar.getAlphabet();
		this.nonterminalMap = new HashMap<String, NonterminalInfo>();
	}
}

rules
@init {	
	for (String nt: NonTerminalCollector.nonterminals)
	{
		NonterminalInfo nI = new NonterminalInfo();
		nI.label = nt;
		this.nonterminalMap.put(nt, nI);
	}
}
	: (^( RULE rule
		{
			// check if entrance nodes are equal for all rule graphs
			NonterminalInfo nI = this.nonterminalMap.get($rule.nonterminal);
			if(nI.typeSequence == null)
			{
				nI.typeSequence = new NodeType[$rule.graph.getExternalCount()];
				for(int i = 0; i < nI.typeSequence.length; i ++)
				{
					nI.typeSequence[i] = $rule.graph.getNodeType($rule.graph.getExternalNode(i));
				}
			}
   
			if(nI.entrance == null)
			{
				nI.entrance = new boolean[$rule.graph.getExternalCount()];
				for(int i = 0; i < nI.typeSequence.length; i ++)
				{
					nI.entrance[i] = !($rule.graph.getNodeTentacle($rule.graph.getExternalNode(i)) instanceof ReductionTentacle || $rule.graph.getNodeTentacle($rule.graph.getExternalNode(i)).getID() == 0 );
				}
			}
			// check the rank and typesequence of graph and nonterminal 
			if(nI.typeSequence.length != $rule.graph.getExternalCount())
				throw new HRGNotTypedNonterminalException(nI.label);
			for(int e = 0; e < nI.typeSequence.length; e++)
			{
				if(nI.typeSequence[e] != $rule.graph.getNodeType($rule.graph.getExternalNode(e)))
					throw new HRGNotTypedNonterminalException(nI.label);
				if(nI.entrance[e] == ($rule.graph.getNodeTentacle($rule.graph.getExternalNode(e)) instanceof ReductionTentacle || $rule.graph.getNodeTentacle($rule.graph.getExternalNode(e)).getID() == 0) )
				{
					throw new HRGNotTypedNonterminalException(nI.label);
				}
			}
 		}
	))*
;

rule returns [String nonterminal, HyperGraph graph]
@init {
	Map<String, NodeInfo> nodemap = new HashMap<String, NodeInfo>();
	
	$graph = new HyperGraph();
}
	: 	^(DEFINITION ^(TYPE name=ID) defnodes=def_tuple[name.getText()]
			{
				$nonterminal = $name.getText();
				int ext = 0;
      			for (NodeInfo n: $defnodes.nodes)
				{ // add all external nodes
					n.id = $graph.newNode(n.type);
					$graph.insertExternal(ext++, n.id);
        			nodemap.put(n.label, n);
					if (n.reduct)
					{
						$graph.setNodeTentacle(n.id, n.type.getReductionTentacle());
					}
				} 
			}
		)
		(
			^(EDGE ^(TYPE name=ID) nodes=edge_tuple[$nonterminal])
			{
				List<NodeInfo> nodelist = new LinkedList<NodeInfo>();
				for (NodeInfo n: $nodes.nodes)
				{ // add all internal nodes
					
					if (nodemap.containsKey(n.label)) n = nodemap.get(n.label);
					else
					{
						n.id = $graph.newNode(n.type);
						nodemap.put(n.label, n);
					}
					nodelist.add(n);
				}
				
				NonterminalInfo nt = this.nonterminalMap.get($name.getText());
				
				if (nt == null)
				{ // nonterminal not yet in nonterminalMap
					Nonterminal nonterm = alphabet.getNonterminal($name.getText());
					if (nonterm != null)
       				{ // but the alphabet already knows about it
						nt = new NonterminalInfo();
						nt.label = nonterm.getLabel();
						nt.entrance = new boolean[nonterm.getRank()];
						nt.typeSequence = new NodeType[nonterm.getRank()];
						for (int i = 0; i < nonterm.getRank(); i++)
						{
							nt.entrance[i] = nonterm.isEntrance(i);
							nt.typeSequence[i] = nonterm.getType(i);
						}
						nt.nonterminal = nonterm;
					}
				}
				if (nt != null)
				{ 
					if (nt.typeSequence == null)
					{ // was in nonterminalMap, but not fully initialized
						nt.typeSequence = new NodeType[nodelist.size()];
						for (int i = 0; i < nodelist.size(); i++)
						{
							nt.typeSequence[i] = nodelist.get(i).type;
						}
							
					}
					else
					{ // was in alphabet, check against current nodelist
						if (nt.typeSequence.length != nodelist.size())
							throw new HRGRankException(nt.label);
						for (int i = 0; i < nt.typeSequence.length; i++)
						{
							if (nt.typeSequence[i] != nodelist.get(i).type)
								throw new HRGNotTypedNonterminalException(nt.label);
						}
					}
					
					int[] nodeArray = new int[nodelist.size()];
					for (int i = 0; i < nodeArray.length; i++)
						nodeArray[i] = nodelist.get(i).id;
       				
					if (nt.nonterminal != null)
					{
        				$graph.addEdge(nt.nonterminal, nodeArray);
          			}
					else
					{
        				for (int i = 0; i < $graph.getNodeTentacle(nodeArray[0]).getSuccessorCount() && i < nodeArray.length; i++)
						{
							$graph.setSuccessorNode(nodeArray[0], i, nodeArray[i]);
            			}
					}
				}
				else
				{
      				int s = nodelist.get(0).type.getSelectorId($name.getText());
					if(s < 0)
					{
						throw new HRGUnknownSelectorException(nodelist.get(0).type, $name.getText());
					}
					$graph.setSuccessorNode(nodelist.get(0).id, s, nodelist.get(1).id);
				}
			}
		)*
;

def_tuple [String nonterminal] returns [List<NodeInfo> nodes]
@init {
	$nodes = new LinkedList<NodeInfo>();
}
	: 	(^(NODE cur=node {
			NodeInfo node = new NodeInfo();
			node.label = $cur.name;
			String type = $cur.type;
			if (type == null) type = NodeTypeCheck.types.get($nonterminal).get($nodes.size());
			node.type = this.alphabet.getNodeType(type);
			if(node.type == null)
				throw new HRGUnknownTypeException(type);
			node.reduct = TentacleReductionCollector.reduction.get($nonterminal).get($nodes.size());
			$nodes.add(node);
		} ))+
;


edge_tuple [String nonterminal] returns [List<NodeInfo> nodes]
@init {
	$nodes = new LinkedList<NodeInfo>();
}
	: 	(^(NODE cur=node {
			NodeInfo node = new NodeInfo();
			node.label = $cur.name;
			String type = $cur.type;
			if (type == null) type = NodeTypeCheck.types.get($nonterminal).get($nodes.size());
			node.type = this.alphabet.getNodeType(type);
			if(node.type == null)
				throw new HRGUnknownTypeException(type);
			$nodes.add(node);
		} ))+
;

node returns [ String name, String type ]
	: 	^(NAME n=ID)
			{ $name = $n.getText(); $type = null; }
	|	^(TYPE t=ID) ^(NAME n=ID)
			{ $name = $n.getText(); $type = $t.getText(); }
	|	NULL
			{ $name = "null"; $type = "Null"; }
;