package de.rwth.aachen.i2.graphreduction.newgrammar;

import java.io.File;
import java.io.IOException;
import java.util.NoSuchElementException;

import org.antlr.runtime.ANTLRFileStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.RecognitionException;
import org.antlr.runtime.tree.CommonTree;
import org.antlr.runtime.tree.CommonTreeNodeStream;

import de.rwth.aachen.i2.graphreduction.antlr.HRGUnknownSelectorException;
import de.rwth.aachen.i2.graphreduction.antlr.HRGUnknownTypeException;
import de.rwth.aachen.i2.graphreduction.grammar.HRGrammar;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.DotExporter;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.NodeTypeCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.GrammarNormalizer;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.NonTerminalCollector;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.ProductivityCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.NodeNameCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.IncreasingCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.RuleConnectedCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.NodeTypeCollector;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.TentacleReductionCollector;
import de.rwth.aachen.i2.graphreduction.newgrammar.observer.TypingCheck;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.ErrorHandler;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.ObserverRegister;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.PhaseHandler;

/**
 * class to be called from outside.
 * 
 * this class puts it all together: give it the {@link HRGrammar} object in the constructor, set some configs ({@link #setDotExport(File, String)} or {@link #setNormalizedGrammarFile(File)}) and call {@link #readGrammar(File)} to load a grammar file into the object.
 * {@link #readGrammar(File)} will then process all steps to collect necessary data, check the grammar and produce the data structures.
 * if anything fails, most probably due to an incorrect grammar, a hopefully helping error message will be emitted through {@link ErrorHandler} class.
 * 
 * in detail, the following steps are done:
 * <ul>
 * <li>create a {@link HRGLexer}</li>
 * <li>create a {@link HRGParser}</li>
 * <li>create a {@link HRGWalkerGrammar}</li>
 * <li>create and register all {@link BaseObserver}</li>
 * <li>process all {@link BaseObserver} with the {@link HRGWalkerGrammar}</li>
 * <li>create and run {@link HRGCollectorGrammar}</li>
 * <li>create and run {@link HRGBuilderGrammar}</li>
 * </ul>
 * After all this, the grammar object should be ready to be used.
 *
 * @author Gereon
 */
public class HRGGrammarReader
{
	private HRGrammar grammar = null;
	
	// configs
	private File normalizedGrammar = null;
	private File dotFolder = null;
	private String dotPrefix = null;
	
	/**
	 * only constructor. give a grammar object to which the grammar filed should be added
	 * @param grammar grammar object
	 */
	public HRGGrammarReader(HRGrammar grammar)
	{
		this.grammar = grammar;
	}
	
	/**
	 * call this method with a valid {@link File} to save a normalized grammar to this file.
	 * call with null, if you do not want to have a normalized grammar somewhere.
	 * the normalized grammar is produced using the {@link GrammarNormalizer} class.
	 * @param normalizedGrammar output file name
	 */
	public void setNormalizedGrammarFile(File normalizedGrammar)
	{
		this.normalizedGrammar = normalizedGrammar;
	}
	
	/**
	 * call this method with a valid {@link File} pointing to a folder and a file prefix to produce dot files for each grammar file that is loaded.
	 * call with null to disable this feature.
	 * the dot files are produced using the {@link DotExporter} class.
	 * @param folder base dot folder
	 * @param fileprefix filename prefix
	 */
	public void setDotExport(File folder, String fileprefix)
	{
		this.dotFolder = folder;
		this.dotPrefix = fileprefix;
	}
	
	/**
	 * read a grammar file and put the content into the {@link HRGrammar} object given in the constructor.
	 * within this method, the whole parsing process is done: lexing, parsing, processing all {@link BaseObserver} objects and filling the {@link HRGrammar} object.
	 * 
	 * if the grammar is malformed in such a way that the parser is unable to recover to any consistent state, a {@link RecognitionException} is thrown here.
	 * @param filename grammar file
	 * @throws IOException if something goes wrong reading the grammar
	 * @throws RecognitionException if something goes wrong in some parser
	 */
	public void readGrammar(File filename) throws IOException, RecognitionException
	{
		// create lexer
		HRGLexer lexer = new HRGLexer(new ANTLRFileStream(filename.getAbsolutePath()));
		CommonTokenStream tokens = new CommonTokenStream(lexer);
	
		// create parser
		HRGParser parser = new HRGParser(tokens);
		CommonTree tree = (CommonTree)parser.rules().getTree();
		CommonTreeNodeStream nodestream = new CommonTreeNodeStream(tree);

		// create treeparser
		HRGWalkerGrammar treeparser = new HRGWalkerGrammar(nodestream);
		PhaseHandler phases = new PhaseHandler(treeparser);
		
		// create and register observers
		ObserverRegister.register(new NodeTypeCheck(treeparser));
		ObserverRegister.register(new NodeNameCheck(treeparser));
		ObserverRegister.register(new RuleConnectedCheck(treeparser));
		ObserverRegister.register(new NodeTypeCollector(treeparser));
		ObserverRegister.register(new NonTerminalCollector(treeparser));
		ObserverRegister.register(new ProductivityCheck(treeparser));
		ObserverRegister.register(new TentacleReductionCollector(treeparser));
		ObserverRegister.register(new IncreasingCheck(treeparser));
		ObserverRegister.register(new TypingCheck(treeparser));
		if (this.dotFolder != null) ObserverRegister.register(new DotExporter(treeparser, this.dotFolder, this.dotPrefix));
		if (this.normalizedGrammar != null) ObserverRegister.register(new GrammarNormalizer(treeparser, this.normalizedGrammar));
		
		try
		{
			phases.prepare();
		}
		catch (NoSuchElementException e)
		{
			e.printStackTrace();
			return;
		}
		
		// process observers
		while (!phases.isDone())
		{
			nodestream.reset();
			treeparser.setObserver(phases);
			if (!treeparser.rules().res)
			{
				ErrorHandler.emitError("Something went wrong while processing " + phases.getActiveObservers() + ", see error messages above.");
				return;
			}
		}

		try
		{
			// two phased grammar building
			nodestream.reset();
			HRGCollectorGrammar collector = new HRGCollectorGrammar(nodestream, this.grammar);
			collector.rules();
		
			nodestream.reset();
			HRGBuilderGrammar builder = new HRGBuilderGrammar(nodestream, this.grammar, collector.nonterminalMap);
			builder.rules();
		}
		catch (HRGUnknownTypeException e)
		{
			ErrorHandler.emitError("A node type was unknown, most probably the given grammar does not match the given java class files. the stack trace is given below.");
			e.printStackTrace();
		}
		catch (HRGUnknownSelectorException e)
		{
			ErrorHandler.emitError("A selector name was unknown, most probably the given grammar does not match the given java class files. the stack trace is given below.");
			e.printStackTrace();
		}
	}
}
