package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;

/**
 * writes all rules to dot files
 * 
 * the class is given a base folder and a filename prefix in the constructor.
 * files are named prefix_nonterminal_number.dot and have to be rendered manually.
 * to render them, use neato.
 * @author Gereon
 */
public class DotExporter extends BaseObserver
{
	// write dot file to this writer
	private BufferedWriter output = null;
	// filename prefix
	private String prefix = null;
	// base folder
	private File folder = null;
	
	// temporary node storage
	private Set<String> nodes = null;
	// edge name counter
	private Map<String, Integer> edges = null;
	
	// count the rules to create file names
	private int num = 1;
	
	/**
	 * create a new {@link DotExporter}
	 * @param parser parser object
	 * @param folder base folder
	 * @param prefix filename prefix
	 */
	public DotExporter(TreeParser parser, File folder, String prefix)
	{
		super(parser);
		this.folder = folder;
		this.prefix = prefix;
		
		this.nodes = new HashSet<String>();
		this.edges = new HashMap<String, Integer>();
		
		this.addDependency("NonTerminalCollector");
		this.addDependency("TentacleReductionCollector");
	}
	
	@Override
	public boolean init()
	{
		// check folder
		if (this.folder.exists())
		{
			// if it is no folder, delete it. hope it isn't something important...
			if (!this.folder.isDirectory()) this.folder.delete();
			else
			{
				// remove all files within the folder. hope again...
				for (File f: this.folder.listFiles())
				{
					if (f.getName().startsWith(this.prefix)) f.delete(); 
				}
			}
		}
		// if there is no folder, create one
		if (!this.folder.exists()) this.folder.mkdir();
		
		return true;
	}

	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		try 
		{
			File file;
			// initialize file
			file = new File(this.folder + "/" + this.prefix + "_" + nonterminal.getText() + "_" + (this.num++) + ".dot");
			this.output = new BufferedWriter(new FileWriter(file));
			this.output.write("digraph " + nonterminal.getText() + "_" + this.num + " {\n");
			
			// clear temporary storage
			this.nodes.clear();
			this.edges.clear();
			
			int i = 0;
			for (Definition def: nodes)
			{ // for each external node:
				// create a node
				this.output.write(def.getName() + " [label=\"" + def + "\", shape=\"doublecircle\"];\n");
				// create an outer node to hold the tentacle
				this.output.write(def.getName() + "_outer [shape=\"point\"];\n");
				if (TentacleReductionCollector.reduction.get(nonterminal.getText()).get(i))
				{ // create reduction or non reduction tentacle
					this.output.write(def.getName() + "_outer -> " + def.getName() + " [arrowhead=\"tee\"];\n");
				}
				else this.output.write(def.getName() + "_outer -> " + def.getName() + " [arrowhead=\"none\"];\n");
				// add to node storage
				this.nodes.add(def.getName());
				i++;
			}
		}
		catch (IOException e) 
		{
			e.printStackTrace();
		}
		return true;
	}
	
	protected boolean ruleEnd(Token nonterminal)
	{
		// close file
		try 
		{
			this.output.write("};\n");
			this.output.flush();
			this.output.close();
		}
		catch (IOException e) {}
		return true;
	}
	
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		String edge = edgename.getText();
		// check of edge already occurred to avoid a name clash for nonterminals 
		int num = 1;
		if (this.edges.containsKey(edge)) num = this.edges.get(edge) + 1; 
		this.edges.put(edge, num);
		String dotname = edge + "_" + num;
		
		try 
		{			
			for (Definition def: nodes)
			{ // create nodes
				if (this.nodes.contains(def.getName())) continue;
				this.output.write(def.getName() + " [label=\"" + def + "\"];\n");
				this.nodes.add(def.getName());
			}
			if (NonTerminalCollector.terminals.contains(edge))
			{ // create a terminal edge 
				Definition from = nodes.get(0);
				Definition to = nodes.get(1);
				this.output.write(from.name.getText() + " -> " + to.name.getText() + " [len=2,label=\"" + edge + "\"];\n");
			}
			else
			{ // create a nonterminal edge
				this.output.write(dotname + " [shape=box];\n");
				int i = 1;
				for (Definition def: nodes)
				{
					if (TentacleReductionCollector.reduction.get(edge).get(i-1))
					{
						this.output.write(def.getName() + " -> " + dotname + " [len=2,arrowhead=\"tee\",label=\"" + i + "\"];\n");
					}
					else
					{
						this.output.write(def.getName() + " -> " + dotname + " [len=2,arrowhead=\"none\",label=\"" + i + "\"];\n");
					}
					i++;
				}
			}
		}
		catch (IOException e) {}
		return true;
	}
	
	public boolean validate()
	{ // nothing to do here...
		return true;
	}
}
