package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;

/**
 * prints a normalized version of the grammar to a file.
 * 
 * the destination filename is given in the constructor.
 * the normalization does not depend on the semantics, so it will always be done, wether the grammar is valid or not. 
 *
 * @author Gereon
 */
public class GrammarNormalizer extends BaseObserver
{
	private FileWriter fileout = null;
	private File filename = null;

	/**
	 * create a new {@link GrammarNormalizer}
	 * @param parser parser object
	 * @param filename destination filename
	 */
	public GrammarNormalizer(TreeParser parser, File filename)
	{
		super(parser);
		this.filename = filename;
	}
	
	private void printNodeList(List<Definition> nodes) throws IOException
	{
		this.fileout.write("(");
		for (int i=0; i<nodes.size(); i++)
		{
			Definition def = nodes.get(i);
			if (i > 0) this.fileout.write(", ");
			if (def.name != null)
			{
				this.fileout.write(def.name.getText());
				if (def.type != null) this.fileout.write(" " + def.type.getText());
			}
			else this.fileout.write("null");
		}
		this.fileout.write(")");
	}

	@Override
	public boolean init()
	{
		try
		{
			this.fileout = new FileWriter(filename);
			return true;
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		return false;
	}

	@Override
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		try
		{
			this.fileout.write(nonterminal.getText() + ": ");
			this.printNodeList(nodes);
			this.fileout.write("\n");
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		return true;
	}

	@Override
	protected boolean ruleEnd(Token nonterminal)
	{
		try
		{
			this.fileout.write("\n");
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		return true;
	}

	@Override
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		try
		{
			this.fileout.write("\t-> " + edgename.getText() + " ");
			this.printNodeList(nodes);
			this.fileout.write("\n");
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		return true;
	}

	@Override
	public boolean validate()
	{
		try
		{
			this.fileout.close();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		return true;
	}
}
