package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.ErrorHandler;

/**
 * collects node types for each rule and checks for consistency
 * 
 * as a node type can be denoted multiple times within a rule, this rule collects the node type for each node in every rule.
 * an error is thrown if some node can not be assigned to some type, because none was given or because multiple conflicting types were given.
 *
 * @author Gereon
 */
public class NodeTypeCollector extends BaseObserver
{
	/**
	 * for every rule: map node names to node types
	 */
	public static List<Map<String, String>> nodetypes = null;
	
	/**
	 * maps node names to node types for the current rule
	 */
	private Map<String, String> types = null;
	
	/**
	 * create a new {@link NodeTypeCollector}
	 * @param parser parser object
	 */
	public NodeTypeCollector(TreeParser parser)
	{
		super(parser);
		this.addDependency("NodeNameCheck");
	}


	@Override
	public boolean init() {
		NodeTypeCollector.nodetypes = new LinkedList<Map<String,String>>();
		return true;
	}

	@Override
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		this.types = new HashMap<String, String>();
		this.edge(nonterminal, nodes);
		return true;
	}

	@Override
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		boolean result = true;
		for (Definition def: nodes)
		{
			if (def.name == null) continue;
			if (def.type == null)
			{
				if (!this.types.containsKey(def.getName()))
					this.types.put(def.getName(), null);
			}
			else if (this.types.get(def.getName()) != null)
			{
				String type = this.types.get(def.getName());
				if (!type.equals(def.type.getText()))
				{
					result = false;
					ErrorHandler.emitError(edgename, "Node " + def.name.getText() + " has type \"" + def.type.getText() + "\" but was previously declared with type \"" + type + "\"");					
				}
			}
			else this.types.put(def.name.getText(), def.type.getText());
		}
		return result;
	}

	@Override
	protected boolean ruleEnd(Token nonterminal)
	{
		boolean res = true;
		for (String node: this.types.keySet())
		{
			if (this.types.get(node) == null)
			{
				ErrorHandler.emitError(nonterminal, "Rule contained node \"" + node + "\" but not type was given.");
				res = false;
			}
		}
		NodeTypeCollector.nodetypes.add(this.types);
		return res;
	}

	@Override
	public boolean validate() {
		return true;
	}
}
