package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.DependencyScheduler;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.ErrorHandler;

/**
 * checks all nonterminals for productivity.
 * 
 * while it uses the very same logic as the {@link DependencyScheduler}, it implements its own simple solver.
 *
 * @author Gereon
 */
public class ProductivityCheck extends BaseObserver
{
	protected Set<String> productive = null;
	protected Set<String> unproductive = null;
	
	/**
	 * save dependencies for each nonterminal
	 * 
	 * for a specific nonterminal, we have Set<Set<String>>
	 * this holds a set of rules that in turn hold a set of edges, the nonterminal depends on
	 */
	private Map<String, Set<Set<String>>> dependencies = null;
	private Set<String> curEdges = null;

	/**
	 * create a new {@link ProductivityCheck}
	 * @param parser parser object
	 */
	public ProductivityCheck(TreeParser parser)
	{
		super(parser);
		this.productive = new HashSet<String>();
		this.unproductive = new HashSet<String>();
		
		this.dependencies = new HashMap<String, Set<Set<String>>>();

		this.addDependency("NonTerminalCollector");
	}
	
	@Override
	public boolean init()
	{
		this.productive = new HashSet<String>(NonTerminalCollector.terminals);
		return true;
	}

	@Override
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		if (!this.dependencies.containsKey(this.curRule.getText()))
		{
			this.dependencies.put(this.curRule.getText(), new HashSet<Set<String>>());
		}
		this.curEdges = new HashSet<String>();
		return true;
	}
	
	protected boolean ruleEnd(Token nonterminal)
	{
		this.dependencies.get(this.curRule.getText()).add(this.curEdges);
		return true;
	}
	
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		this.curEdges.add(edgename.getText());
		return true;
	}

	@Override
	public boolean validate()
	{
		boolean changed = true; // at least one run...
		
		/* 
		 * iteratively try to resolve dependencies, mark resolved nonterminals as productive
		 * at the beginning, only terminals are productive
		 */
		while (changed)
		{
			changed = false;
			
			// we'll check all nonterminals and try to find any productive ones
			// use an iterator to remove something later, otherwise HashMap will throw a ConcurrentModificationException 
			Iterator<String> iterator = this.dependencies.keySet().iterator();
			while (iterator.hasNext())
			{
				String nonterminal = iterator.next();
				Set<Set<String>> dep = this.dependencies.get(nonterminal);
				
				// we'll iterate over all rules for this nonterminal and try to find productive ones
				for (Set<String> d: dep)
				{
					boolean resolved = true;
					// check if all edges are productive
					for (String s: d) resolved = resolved && this.productive.contains(s);
					if (resolved)
					{
						this.productive.add(nonterminal);
						iterator.remove();
						changed = true; // we changed something
						break; // continue with next nonterminal
					}
				}
			}
		}
		
		this.unproductive = this.dependencies.keySet();
		if (!this.unproductive.isEmpty())
		{
			ErrorHandler.emitError("There are unproductive nonterminals: " + this.unproductive.toString());
			return false;
		}
		return true;
	}
}
 