package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.InferenceException;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.InferenceInformation;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.InferenceObject;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.InferenceSolver;

/**
 * identifies reduction tentacles and nonreduction tentacles.
 * 
 * these results are stored in the static {@link #reduction} variable.
 * a tentacle is a reduction tentacle, if it is not possible to create a terminal edge from the external node into the nonterminal.
 * 
 * the identification of reduction tentacles is based on an inference system. we start with some simple properties that enforce a reduction tentacle.
 * afterwards, we apply inference rules until everything is done.
 * 
 * to infer this, we have the following rules:add
 * 
 * a tentacle T with associated external node N is no reduction tentacle, if
 * <ul>
 * <li>T can produce a terminal edge from N</li>
 * <li>T can produce a nonterminal that is connected to N by a non-reduction tentacle</li>
 * </li>
 * 
 * if T can not do one of the above, T is a reduction tentacle. 
 *
 * @author Gereon
 */
public class TentacleReductionCollector extends BaseObserver
{
	/**
	 * contains information for each inference
	 *
	 * @author Gereon
	 */
	public class TentacleInformation extends InferenceInformation
	{
		/**
		 * type of inference: true => true or false => false
		 */
		public boolean ifthen = false;
		/**
		 * constructor
		 * @param ifthen specifies the type of inference
		 */
		public TentacleInformation(boolean ifthen)
		{
			this.ifthen = ifthen;
		}
		public boolean equals(Object obj)
		{
			TentacleInformation o = (TentacleInformation) obj;
			if (o == null) return false;
			return o.ifthen == this.ifthen;
		}
		public String toString()
		{
			if (this.ifthen) return "pos";
			else return "neg";
		}
	}
	
	/**
	 * this class represents a tentacle to infer if it is a reduction tentacle
	 *
	 * @author Gereon
	 */
	public class Tentacle extends InferenceObject<Tentacle, TentacleInformation>
	{
		/**
		 * nonterminal of this tentacle
		 */
		public String nonterminal = null;
		/**
		 * selector of the nonterminal
		 */
		public int selector = 0;
		/**
		 * is it a reduction tentacle?
		 */
		public boolean reduction = false;
		
		/**
		 * the constructor
		 * @param nonterminal nonterminal name
		 * @param selector selector number
		 */
		public Tentacle(String nonterminal, int selector)
		{
			this.nonterminal = nonterminal;
			this.selector = selector;
		}

		@Override
		public boolean inferFrom(Tentacle condition, TentacleInformation information) throws InferenceException
		{	
			this.infered = true;
			if (information.ifthen && condition.reduction)
			{
				this.reduction = true;
				return true;
			}
			else if ((!information.ifthen) && (!condition.reduction))
			{
				this.reduction = false;
				return true;
			}
			else return false;
		}

		@Override
		public void checkInference(Tentacle condition, TentacleInformation information) throws InferenceException
		{
			if (information.ifthen && condition.reduction)
			{
				if (!this.reduction) throw new InferenceException("Tentacle " + this + " was computed to be a non-reduction tentacle, but the current computation says it is.");
			}
			else if ((!information.ifthen) && (!condition.reduction))
			{
				if (this.reduction) throw new InferenceException("Tentacle " + this + " was computed to be a reduction tentacle, but the current computation says it is not.");
			}
		}
		
		public boolean equals(Object o)
		{
			Tentacle obj = (Tentacle)o;
			if (obj == null) return false;
			return (obj.nonterminal.equals(this.nonterminal)) && (obj.selector == this.selector);
		}
		
		public String toString()
		{
			return this.nonterminal + ":" + this.selector + "[" + this.reduction + "]";
		}

		@Override
		public void copyFrom(Tentacle obj)
		{
			this.infered = obj.infered;
			this.nonterminal = obj.nonterminal;
			this.reduction = obj.reduction;
			this.selector = obj.selector;
		}
	}

	/**
	 * maps nonterminal names to a list of booleans.
	 * if the i'th boolean is true, the i'th tentacle of this nonterminal is a reduction tentacle.
	 */
	public static Map<String,Vector<Boolean>> reduction = null;
	
	/**
	 * our solver...
	 */
	private InferenceSolver<Tentacle, TentacleInformation> solver = null;
	
	/**
	 * maps the node name to the node number of the nonterminal in the current rule
	 */
	private Map<String, Integer> curMapping = null;
	
	private Map<String, Tentacle> curReductionCandidates = null;
	
	/**
	 * the constructor
	 * @param parser tree parser
	 */
	public TentacleReductionCollector(TreeParser parser) {
		super(parser);
		this.addDependency("NonTerminalCollector");
		this.addDependency("NodeTypeCollector");
		this.addDependency("ProductivityCheck");
		TentacleReductionCollector.reduction = new HashMap<String, Vector<Boolean>>();
	}
	
	private void putCandidate(Tentacle candidate)
	{
		this.curReductionCandidates.put(candidate.toString(), candidate);
	}
	private void removeCandidate(Tentacle candidate)
	{
		this.curReductionCandidates.remove(candidate.toString());
	}
	
	public boolean init()
	{
		TentacleReductionCollector.reduction.clear();
		this.solver = new InferenceSolver<Tentacle, TentacleInformation>();
		return true;
	}

	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		this.curMapping = new HashMap<String, Integer>();
		this.curReductionCandidates = new HashMap<String, Tentacle>();
		for (int i=0; i<nodes.size(); i++)
		{
			this.curMapping.put(nodes.get(i).getName(), new Integer(i));
			this.putCandidate(new Tentacle(nonterminal.getText(), i));
		}
		return true;
	}
	
	protected boolean ruleEnd(Token nonterminal)
	{
		for (Tentacle reductionTentacle: this.curReductionCandidates.values())
		{
			reductionTentacle.reduction = true;
			reductionTentacle.infered = true;
			this.solver.add(null, reductionTentacle, new TentacleInformation(true));
		}
		this.curMapping.clear();
		this.curMapping = null;
		return true;
	}

	public boolean edge(Token edgename, List<Definition> nodes)
	{
		String edge = edgename.getText();
		String rule = this.curRule.getText();
		if (NonTerminalCollector.terminals.contains(edge))
		{ // is a terminal edge
			if (this.curMapping.containsKey(nodes.get(0).getName()))
			{
				// direct fact for node 0
				int external = this.curMapping.get(nodes.get(0).getName());
				this.solver.add(null, new Tentacle(edge, 0), null);
				this.solver.add(new Tentacle(edge, 0), new Tentacle(rule, external), new TentacleInformation(false));
				this.removeCandidate(new Tentacle(this.curRule.getText(), external));
			}
		}
		else
		{
			for (int i=0; i<nodes.size(); i++)
			{
				// is it an external node?
				if (!this.curMapping.containsKey(nodes.get(i).getName())) continue;
				int external = this.curMapping.get(nodes.get(i).getName());
				// infer from embedded nonterminals
				this.solver.add(new Tentacle(edge, i), new Tentacle(rule, external), new TentacleInformation(false));
				this.solver.add(new Tentacle(rule, external), new Tentacle(edge, i), new TentacleInformation(true));
				// no direct facts
				this.removeCandidate(new Tentacle(this.curRule.getText(), i));
			}
		}
		return true;
	}
	
	public boolean validate() 
	{
		try
		{
			this.solver.solve();
			
			for (Tentacle t: this.solver.getObjects())
			{
				if (t == null) continue;
				if (NonTerminalCollector.terminals.contains(t.nonterminal)) continue;
				if (!TentacleReductionCollector.reduction.containsKey(t.nonterminal))
				{
					TentacleReductionCollector.reduction.put(t.nonterminal, new Vector<Boolean>());
				}
				for (int i=TentacleReductionCollector.reduction.get(t.nonterminal).size(); i<t.selector+1; i++)
				{
					TentacleReductionCollector.reduction.get(t.nonterminal).add(i, true);
				}
				TentacleReductionCollector.reduction.get(t.nonterminal).set(t.selector, t.reduction);
			}
			return true;
		}
		catch (InferenceException e)
		{
			e.printStackTrace();
			return false;
		}
	}
}
