package de.rwth.aachen.i2.graphreduction.newgrammar.observer;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

import de.rwth.aachen.i2.graphreduction.newgrammar.utils.BaseObserver;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.Definition;
import de.rwth.aachen.i2.graphreduction.newgrammar.utils.ErrorHandler;

/**
 * checks the correct (edge) typing of a heap abstraction grammar
 * 
 * this class checks every node in every rule for outgoing edges.
 * an error is thrown if
 * <ul>
 * <li>a node has two edges he is connected to via nonreduction tentacles</li>
 * <li>a node is connected to a nonreduction tentacle and one or more terminal edges</li>
 * <li>two nodes have a different set of outgoing terminal edges</li>
 * </ul>
 * 
 * @author Gereon
 */
public class TypingCheck extends BaseObserver
{
	private class OutgoingEdges
	{
		public String node;
		public String nonterminal = null;
		public Set<String> terminals = new HashSet<String>();
		
		public OutgoingEdges(String node)
		{
			this.node = node;
		}
		
		public boolean addTerminal(Token edge)
		{
			if (this.nonterminal != null)
			{
				ErrorHandler.emitError(edge, "Node \"" + this.node + "\" was already connected to non reduction nonterminal \"" + this.nonterminal + "\", but is now connected to the terminal \"" + edge.getText() + "\".");
				return false;
			}
			else if (this.terminals.contains(edge.getText()))
			{
				ErrorHandler.emitError(edge, "Node \"" + this.node + "\" is declared to have an outgoing terminal edge \"" + edge.getText() + "\" but this node already had this outgoing terminal edge within the current rule.");
				return false;
			}
			else this.terminals.add(edge.getText());
			return true;
		}
		
		public boolean addNonterminal(Token edge)
		{
			if (this.terminals.size() > 0)
			{
				ErrorHandler.emitError(edge, "Node \"" + this.node + "\" was already connected to at least one terminal, but is now connected to the nonterminal \"" + edge.getText() + "\".");
				return false;
			}
			else if (this.nonterminal != null)
			{
				ErrorHandler.emitError(edge, "Node \"" + this.node + "\" was already connected to nonterminal \"" + this.nonterminal + "\", but is now connected to another nonterminal \"" + edge.getText() + "\".");
				return false;
			}
			else this.nonterminal = edge.getText();
			return true;
		}
		
		public boolean isCompatible(OutgoingEdges o)
		{		
			if (this.nonterminal == null)
			{
				if (o.nonterminal != null) return true;
				return this.terminals.equals(o.terminals);
			}
			else
			{ // 
				return true;
			}
		}
		
		/**
		 * merge this object with another one, make the most "concrete" edges from both
		 * @param o other object
		 * @return true if they are compatible, false otherwise
		 */
		public boolean merge(OutgoingEdges o)
		{
			if (! this.isCompatible(o)) return false;
			
			if (this.nonterminal == null) return true;
			if (o.nonterminal != null) return true;
			this.nonterminal = null;
			this.terminals.addAll(o.terminals);
			
			return true;
		}
		
		public String toString()
		{
			return "( " + this.nonterminal + ", " + this.terminals + " )";
		}
	}
	/**
	 * maps node names to outgoing edges 
	 */
	private Map<String, OutgoingEdges> current = null;
	
	/**
	 * maps type names to outgoing edges
	 */
	private Map<String, OutgoingEdges> types = null;

	private Map<String, String> typeMapping = null;

	/**
	 * the constructor...
	 * @param parser
	 */
	public TypingCheck(TreeParser parser)
	{
		super(parser);
		this.addDependency("NonTerminalCollector");
		this.addDependency("TentacleReductionCollector");
		this.addDependency("NodeTypeCollector");
	}

	@Override
	public boolean init()
	{
		this.types = new HashMap<String, OutgoingEdges>();
		this.ruleNumber = 0;
		return true;
	}

	@Override
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		this.current = new HashMap<String, OutgoingEdges>();
		this.typeMapping = NodeTypeCollector.nodetypes.get(this.ruleNumber);
		return true;
	}

	@Override
	protected boolean ruleEnd(Token nonterminal)
	{
		boolean res = true; 
		for (String node: this.current.keySet())
		{
			String type = this.typeMapping.get(node);
			if (this.types.containsKey(type))
			{
				if (!this.types.get(type).isCompatible(this.current.get(node)))
				{
					ErrorHandler.emitError(nonterminal, "Outgoing edges for node \"" + node + "\" are incompatible with those of its type \"" + type + "\"");
					ErrorHandler.emitError("Outgoing for node: " + this.current.get(node));
					ErrorHandler.emitError("Outgoing for type: " + this.types.get(type));
					res = false;
				}
				else if (!this.types.get(type).merge(this.current.get(node)))
				{
					ErrorHandler.emitError("This should not happen!");
				}
			}
			else
			{
				this.types.put(type, this.current.get(node));
			}
		}
		return res;
	}

	@Override
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		boolean res = true;
		String edge = edgename.getText();
		if (NonTerminalCollector.terminals.contains(edge))
		{
			String node = nodes.get(0).getName();

			if (!this.current.containsKey(node))
			{
				this.current.put(node, new OutgoingEdges(node));
			}
			res &= this.current.get(node).addTerminal(edgename);
		}
		else
		{
			Vector<Boolean> reduction = TentacleReductionCollector.reduction.get(edge);
			for (int i = 0; i < reduction.size(); i++)
			{
				if (!reduction.get(i))
				{
					Definition def = nodes.get(i);
					String node = def.getName();
					if (!this.current.containsKey(node))
					{
						this.current.put(node, new OutgoingEdges(node));
					}
					res &= this.current.get(node).addNonterminal(edgename);
				}
			}
		}
		return res;
	}

	@Override
	public boolean validate()
	{
		return true;
	}

}
