package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

/**
 * base class for all checks, collectors, etc.
 * 
 * it defines an interface that is meant to be used directly in the grammar.
 * the protected methods should be overridden by inheriting classes to do something useful.
 * 
 * an inheriting class can be reasonably sure that {@link #init()} will be called only once at the beginning of a run.
 * afterwards, a number of rules are processed. each rule is started with {@link #ruleStart(Token, List)} and finished with {@link #ruleEnd(Token)}.
 * {@link #edge(Token, List)} will only occur within those two calls.
 * after all rules are processed, a single {@link #validate()} allows final checks.
 * 
 * all processing methods, those are {@link #init()}, {@link #edge(Token, List)}, {@link #startRule(Token, List)}, {@link #edge(Token, List)}, {@link #validate()}
 * return a boolean that indicates the success of the specific action.
 * false indicates a severe error in the grammar, that prevents further evaluation and thereby forcing the parser to abort after the current run.
 * an error (i.e. a return value of false) usually states, that there is an error within the grammar, not in the parser (unless explicitly stated in the error message). 
 * 
 * the output data of a {@link BaseObserver} that returned false will probably contain no meaningful information afterwards!
 * 
 * a {@link BaseObserver} can depend on another. this should be done (and only done) if you need some data that is collected or calculated by this other observer.
 * be aware that the dependent observer cannot access the data from the observer it depends on in the constructor.
 * this data will only be available in the initialization ({@link #init()}) and later. 
 *
 * @author Gereon
 */
public abstract class BaseObserver implements DependencyObject
{
	/**
	 * {@link Token} of the nonterminal name of the current rule
	 */
	protected Token curRule = null;
	/**
	 * the number of the current rule
	 */
	protected int ruleNumber = 0;
	/**
	 * the {@link TreeParser} that processes the tree and calls this observer
	 */
	protected TreeParser parser = null;
	
	/**
	 * the dependencies
	 */
	protected Set<String> dependencies = null;
	protected Set<DependencyObject> realDependencies = null;
	
	/**
	 * the only constructor.
	 * 
	 * note that any data from {@link BaseObserver} that this class is dependent on is not accessible here.
	 * do such initialization stuff in init(). 
	 * @param parser the {@link TreeParser} that will call this observer
	 */
	public BaseObserver(TreeParser parser)
	{
		this.parser = parser;
		this.dependencies = new HashSet<String>();
	}
	
	/**
	 * add another observer to the set of dependencies. 
	 * @param name class name of the dependency
	 */
	protected void addDependency(String name)
	{
		this.dependencies.add(name);
	}
	
	/**
	 * return set of dependencies.
	 * when this method is called, all class names added by {@link #addDependency(String)} are resolved and cached.
	 * later calls to {@link #addDependency(String)} are not forbidden but will not have any effect.
	 * @return dependencies
	 */
	public Set<DependencyObject> getDependencies()
	{
		if (this.realDependencies == null)
		{
			this.realDependencies = new HashSet<DependencyObject>();
			for (String name: this.dependencies)
			{
				this.realDependencies.add(ObserverRegister.getObserver(name));
			}
		}
		return this.realDependencies;
	}
	
	/**
	 * to be called by the parser upon a new rule
	 * @param nonterminal {@link Token} of the nonterminal
	 * @param nodes {@link List} of {@link Definition} specifying the outer nodes
	 * @return false on error, true otherwise
	 */
	public boolean startRule(Token nonterminal, List<Definition> nodes)
	{
		this.curRule = nonterminal;
		return this.ruleStart(nonterminal, nodes);
	}
	
	/**
	 * to be called by the parser upon the end of a rule
	 * @param nonterminal {@link Token} of the nonterminal
	 * @return false on error, true otherwise
	 */
	public boolean endRule(Token nonterminal)
	{
		boolean res = this.ruleEnd(nonterminal);
		this.ruleNumber++;
		return res;
	}
	
	/**
	 * to be called before parsing the tree to initialize stuff, i.e. fetch data from other {@link BaseObserver}.
	 * note that those must be added as dependencies within the constructor.
	 * @return false on error, true otherwise
	 */
	public abstract boolean init();
	
	/**
	 * internal method, to be called by {@link #startRule(Token, List)}
	 * @param nonterminal Token of nonterminal
	 * @param nodes List of nodes
	 * @return false on error, true otherwise
	 */
	protected abstract boolean ruleStart(Token nonterminal, List<Definition> nodes);
	
	/**
	 * to be called by the parser when the current rule is finished
	 * @param nonterminal {@link Token} of the rules nonterminal
	 * @return false on error, true otherwise
	 */
	protected abstract boolean ruleEnd(Token nonterminal);
	
	/**
	 * to be called by the parser upon an edge
	 * @param edgename {@link Token} of the edge name
	 * @param nodes {@link List} of {@link Definition} specifying the edges outer nodes
	 * @return false on error, true otherwise
	 */
	public abstract boolean edge(Token edgename, List<Definition> nodes);
	
	/**
	 * to be called when parsing is done.
	 * in some cases, checks are only possible after all data has been collected...
	 * @return false on error, true otherwise
	 */
	public abstract boolean validate();
	
	/**
	 * return simple class name as String
	 */
	public String toString()
	{
		return this.getClass().getSimpleName();
	}
}
