package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

/**
 * a generic class to solve dependency requirements.
 * 
 * add a couple of dependencies using the {@link #add(DependencyObject)} or {@link #addAll(Collection)} method and call {@link #reset()}.
 * then, this class can be used as an {@link Iterator} over {@link Set} objects that contain {@link DependencyObject} objects.
 * 
 * @author Gereon
 * @param <T> type of the dependency objects 
 */
public class DependencyScheduler<T extends DependencyObject> implements Iterable<Set<T>>, Iterator<Set<T>>
{
	
	private Map<T, Set<DependencyObject>> dependencies = null;
	private Map<T, Set<DependencyObject>> remaining = null;
	
	/**
	 * the constructor
	 */
	public DependencyScheduler()
	{
		this.dependencies = new HashMap<T, Set<DependencyObject>>();
		this.remaining = new HashMap<T, Set<DependencyObject>>();
	}
	
	/**
	 * add a new {@link DependencyObject} to the schedule
	 * @param object new object
	 * @return false, if object was already in schedule, true otherwise
	 */
	public boolean add(T object)
	{
		if (this.dependencies.containsKey(object)) return false;
		this.dependencies.put(object, object.getDependencies());
		return true;
	}
	
	/**
	 * add all new {@link DependencyObject} to the schedule
	 * @param objects collection of new objects
	 * @return false, if some object was already in schedule, true otherwise
	 */
	public boolean addAll(Collection<T> objects)
	{
		boolean res = true;
		for (T obj: objects) res = this.add(obj) && res;
		return res;
	}
	
	/**
	 * remove all dependencies
	 */
	public void clear()
	{
		this.dependencies.clear();
		this.remaining.clear();
	}
	
	/**
	 * reset current schedule run, initialize to start a new schedule
	 */
	public void reset()
	{
		this.remaining.clear();
		for (T obj: this.dependencies.keySet())
		{
			Set<DependencyObject> data = new HashSet<DependencyObject>();
			data.addAll(this.dependencies.get(obj));
			this.remaining.put(obj, data);
		}
	}
	
	/**
	 * return an iterator over the schedule
	 */
	public Iterator<Set<T>> iterator()
	{
		return this;
	}
	
	/**
	 * check if there are iterations left to do
	 */
	public boolean hasNext()
	{
		return !this.remaining.isEmpty();
	}
	
	/**
	 * get next set of objects
	 */
	public Set<T> next()
	{
		Set<T> cur = new HashSet<T>();
		
		for (T obj: this.remaining.keySet())
		{
			if (this.remaining.get(obj).isEmpty()) cur.add(obj);
		}
		
		if (cur.isEmpty())
		{
			String e = "The given dependencies could not be resolved. The following objects and dependencies remain unmet.\n";
			e += "\tObjects: " + this.remaining.keySet() + "\n";
			e += "\tDependencies:\n";
			for (T obj: this.remaining.keySet())
			{
				e += "\t\t" + obj.toString() + ": " + dependencies.get(obj) + "\n";
			}
			
			throw new NoSuchElementException(e);
		}
		
		for (T obj: cur)
		{
			this.remaining.remove(obj);
			for (T o: this.remaining.keySet())
			{
				this.remaining.get(o).remove(obj);
			}
		}
		
		return cur;
	}
	
	/**
	 * will throw an {@link UnsupportedOperationException}.
	 */
	public void remove()
	{
		throw new UnsupportedOperationException();
	}
	
	/**
	 * dump the dependency graph to a dot file.
	 * as long as dependencies can be resolved, nodes are ranked.
	 * render with "dot" (not "neato" etc.) to get a nice picture...
	 * 
	 * attention: calling this method will reset the iterator!
	 * @param filename destination file
	 */
	public void dumpDependencyGraph(File filename)
	{
		this.reset();
		try
		{
			BufferedWriter out = new BufferedWriter(new FileWriter(filename));
			out.write("digraph {\n");
			
			this.reset();
			Set<T> printed = new HashSet<T>();
		
			// print all nodes / objects
			try
			{ // generate based on schedule information
				for (Set<T> objects: this)
				{
					out.write("\t{ rank=same;\n");
					for (T obj: objects)
					{
						out.write("\t\t\"" + obj + "\"\n");
						printed.add(obj);
					}
					out.write("\t}\n");
				}
			}
			catch (NoSuchElementException e)
			{ // print nodes that contain unresolved dependencies...
				for (T obj: this.dependencies.keySet())
				{
					if (printed.contains(obj)) continue;
					out.write("\t\t\"" + obj + "\"\n");
				}
			}
			
			// print all edges / dependencies
			for (T dest: this.dependencies.keySet())
			{
				Set<DependencyObject> deps = this.dependencies.get(dest);
				for (DependencyObject dep: deps)
				{
					out.write("\t\"" + dep + "\" -> \"" + dest + "\"\n");
				}
			}
			
			out.write("}\n");
			out.close();
			
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		this.reset();
	}
}
