package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import org.antlr.runtime.Token;

/**
 * central error handler for the whole new grammar.
 * 
 * all errors (and warnings) are just printed to stderr by {@link #emit(String, Token, String)}.
 * all other methods, like {@link #emitError(String)} or {@link #emitWarning(String)} are only wrappers to the {@link #emit(String, Token, String)} method.
 *
 * @author Gereon
 */
public class ErrorHandler
{
	/**
	 * print error message using the {@link #printError(String)} method.
	 * the given {@link Token} object should be the one where the error occurred or was recognized, one should do something like a "best guess" here.
	 * if the {@link Token} is null, the location information in omitted.
	 * @param type "error", "warning" or something alike
	 * @param position a {@link Token} that describes the position where the error occurred or was recognized
	 * @param msg error string
	 */
	public static void emit(String type, Token position, String msg)
	{
		if (position == null)
		{
			ErrorHandler.printError(type + ": " + msg);
		}
		else
		{
			ErrorHandler.printError(type + " at position " + position.getLine() + ":" + position.getCharPositionInLine());
			ErrorHandler.printError("\t" + msg);
		}
	}
	
	/**
	 * print an error message without position information
	 * @param msg error string
	 */
	public static void emitError(String msg)
	{
		ErrorHandler.emit("Error", null, msg);
	}
	
	/**
	 * print an error message with position information
	 * @param position position token
	 * @param msg error string
	 */
	public static void emitError(Token position, String msg)
	{
		ErrorHandler.emit("Error", position, msg);
	}
	
	/**
	 * print a warning message without position information
	 * @param msg warning string
	 */
	public static void emitWarning(String msg)
	{
		ErrorHandler.emit("Warning", null, msg);
	}
	
	/**
	 * print a warning message with position information
	 * @param position position token
	 * @param msg warning string
	 */
	public static void emitWarning(Token position, String msg)
	{
		ErrorHandler.emit("Warning", position, msg);
	}
	
	/**
	 * print some string to error output, currently stderr.
	 * change this method to redirect error output
	 * @param msg error string
	 */
	protected static void printError(String msg)
	{
		System.err.println(msg);
	}
}
