package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

/**
 * this class can solve simple inference systems.
 * 
 * the system consists of objects (or variables) that want to compute some data based on another object where each inference has some additional information about the computation.
 * the solver takes base objects (those, that to not depend on others, i.e. depend on <code>null</code>) and starts inferring from those objects.
 *
 * @author Gereon
 * @param <T> objects that want to infer data
 * @param <I> additional informations about inferences
 */
 
public class InferenceSolver <T extends InferenceObject<T, I>, I extends InferenceInformation>
{	
	/**
	 * this class is used to store all information in one mapping
	 *
	 * @author Gereon
	 */
	public class Inference
	{
		/**
		 * the consequence
		 */
		public T consequence = null;
		/**
		 * some information about the inference
		 */
		public I information = null;
		
		/**
		 * the constructor
		 * @param consequence consequence data
		 * @param information some information
		 */
		public Inference(T consequence, I information)
		{
			this.consequence = consequence;
			this.information = information;
		}
		
		public boolean equals(Object o)
		{
			if (o == null) return false;
			// I'm pretty sure this will work...
			@SuppressWarnings("unchecked")
			Inference obj = (Inference)o;
			
			if (this.consequence != null)
			{
				if (!this.consequence.equals(obj.consequence)) return false;
			}
			else if (obj.consequence != null) return false;
			
			if (this.information != null)
			{
				if (!this.information.equals(obj.information)) return false;
			}
			else if (obj.information != null) return false;
			
			return true;
		}
		public String toString()
		{
			return "( " + this.consequence + ", " + this.information + " )";
		}
	}
	
	private Set<T> objects = null;
	private Map<T, Set<Inference>> inferences = null;
	
	/**
	 * default constructor
	 */
	public InferenceSolver()
	{
		this.inferences = new HashMap<T, Set<Inference>>();
		this.objects = new HashSet<T>();
	}
	
	private T getEqualObject(T obj)
	{
		if (obj == null) return null;
		for (T o: this.objects)
		{
			if (o == null) continue;
			if (obj.equals(o))
			{
				if ((!o.infered) && obj.infered) o.copyFrom(obj);
				return o;
			}
		}
		return obj;
	}

	/**
	 * add an inference, consisting of the condition object and the inference
	 * @param condition
	 * @param inference
	 */
	public void add(T condition, Inference inference)
	{		
		// prevent object duplication
		condition = this.getEqualObject(condition);

		if (!this.objects.contains(condition)) this.objects.add(condition);
		inference.consequence = this.getEqualObject(inference.consequence);
		if (!this.objects.contains(inference.consequence)) this.objects.add(inference.consequence);

		
		if (!this.inferences.containsKey(condition))
		{
			Set<Inference> set = new HashSet<Inference>();
			set.add(inference);
			this.inferences.put(condition, set);
		}
		else
		{
			boolean found = false;
			for (Inference inf: this.inferences.get(condition))
			{
				if (inference.equals(inf)) found = true;
			}
			if (!found) this.inferences.get(condition).add(inference);
		}
	}
	
	/**
	 * add an inference consisting of the condition object, the consequence object and inference information
	 * @param condition
	 * @param consequence
	 * @param information
	 */
	public void add(T condition, T consequence, I information)
	{
		this.add(condition, new Inference(consequence, information));
	}
	
	/**
	 * solve the inference system.
	 * solve throws an {@link InferenceException} with a (hopefully) helpful error message if there are no objects to start with or not all objects could be inferred.
	 * objects to start with should be added as an inference with <code>null</code> as condition object.
	 * 
	 * @throws InferenceException
	 */
	public void solve() throws InferenceException
	{
		this.dumpInferenceGraph(new File("dot/Inferences.dot"));
		// objects that already are inferred
		Set<T> inferred = new HashSet<T>();
		if (this.inferences.get(null) == null)
		{
			throw new InferenceException("There is no object to start with. add an object with condition \"null\"!");
		}
		for (Inference i: this.inferences.get(null))
		{
			inferred.add(i.consequence);
		}
		
		// objects that are inferred, but inferences with these objects as condition have still to be done
		Queue<T> todo = new LinkedList<T>(inferred);
		
		while (!todo.isEmpty())
		{
			T item = todo.remove();
			
			if (!this.inferences.containsKey(item))
			{
				continue;
			}
			Set<Inference> infs = this.inferences.get(item);
			
			for (Inference inf: infs)
			{
				if (inferred.contains(inf.consequence))
				{
					inf.consequence.checkInference(item, inf.information);
				}
				else
				{
					if (inf.consequence.inferFrom(item, inf.information))
					{
						inferred.add(inf.consequence);
						if (this.inferences.containsKey(inf.consequence)) todo.add(inf.consequence);
					}
				}
			}
		}
		for (T obj: this.inferences.keySet())
		{
			if (obj == null) continue;
			if (!inferred.contains(obj))
			{
				throw new InferenceException("variable " + obj + " could be inferred. the inferred objects are: \n" + inferred.toString() + ", missing: " + this.inferences.toString());
			}
		}
	}
	
	/**
	 * get all objects
	 * @return set of objects
	 */
	public Set<T> getObjects()
	{
		return this.objects;
	}
	
	/**
	 * dump inference graph to a file in dot format
	 * @param filename filename
	 */
	public void dumpInferenceGraph(File filename)
	{
		try
		{
			BufferedWriter out = new BufferedWriter(new FileWriter(filename));
			out.write("digraph {\n");
			
			// print all nodes / objects
			for (T obj: this.objects)
			{
				out.write("\t\"" + obj  + "\"\n");
			}
			
			// print all edges / dependencies
			for (T obj: this.objects)
			{
				if (!this.inferences.containsKey(obj)) continue;
				Set<Inference> infs = this.inferences.get(obj);
				for (Inference inf: infs)
				{
					out.write("\t\"" + obj + "\" -> \"" + inf.consequence + "\" [label=\"" + inf.information + "\", len=2]\n");
				}
			}
			
			out.write("}\n");
			out.close();
			
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
	}
}
