package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

/**
 * combins multiple {@link BaseObserver} into one object.
 * 
 * this class offers a convenient mechanism to handle multiple {@link BaseObserver} instances in one run.
 * it will simply broadcast every call to all child observers. 
 *
 * @author Gereon
 */
public class ObserverHandler extends BaseObserver
{
	/**
	 * set of child observers
	 */
	protected Set<BaseObserver> observers;
	
	/**
	 * constructor
	 * @param parser the {@link TreeParser} that will call this observer
	 */
	public ObserverHandler(TreeParser parser)
	{
		super(parser);
		this.observers = new HashSet<BaseObserver>();
	}

	/**
	 * add another child observer
	 * @param observer child
	 */
	public void add(BaseObserver observer)
	{
		this.observers.add(observer);
	}
	
	/**
	 * add some other child observers
	 * @param observers new childs
	 */
	public void addAll(Set<BaseObserver> observers)
	{
		this.observers.addAll(observers);
	}
	
	/**
	 * get a list of the child observers
	 * @return {@link List} of childs
	 */
	public Set<BaseObserver> getObservers()
	{
		return this.observers;
	}
	
	@Override
	public boolean init()
	{
		boolean result = true;
		for (BaseObserver observer: this.observers)
		{
			if (!observer.init()) result = false;
		}
		return result;
	}
	
	@Override
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		boolean result = true;
		for (BaseObserver observer: this.observers)
		{
			if (!observer.startRule(nonterminal, nodes)) result = false;
		}
		return result;
	}
	
	@Override
	protected boolean ruleEnd(Token nonterminal)
	{
		boolean result = true;
		for (BaseObserver observer: this.observers)
		{
			if (!observer.endRule(nonterminal)) result = false;
		}
		return result;
	}

	@Override
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		boolean result = true;
		for (BaseObserver observer: this.observers)
		{
			if (!observer.edge(edgename, nodes)) result = false;
		}
		return result;
	}
	
	@Override
	public boolean validate()
	{
		boolean result = true;
		for (BaseObserver observer: this.observers)
		{
			if (!observer.validate()) result = false;
		}
		return result;
	}
	
	/**
	 * print a nice output similar to {@link List} using {@link Class#getSimpleName()} of the childs classes
	 */
	@Override
	public String toString()
	{
		String res = "[ ";
		boolean first = true;
		for (BaseObserver ob: this.observers)
		{
			if (!first) res += ", ";
			first = false;
			res += ob.getClass().getSimpleName();
		}
		return res + " ]";
	}
}
