package de.rwth.aachen.i2.graphreduction.newgrammar.utils;

import java.io.File;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;

import org.antlr.runtime.Token;
import org.antlr.runtime.tree.TreeParser;

/**
 * collects all {@link BaseObserver} and executes them according to specified dependencies.
 * 
 * this class is basically a wrapper using {@link DependencyScheduler} to resolve dependencies for {@link BaseObserver} objects.
 * 
 * to begin a new schedule run, call {@link #prepare()}.
 * if cyclic dependencies are encountered, a {@link NoSuchElementException} is thrown with a meaningful error message.
 * if {@link #prepare()} is not called, a {@link NullPointerException} will be thrown at some point.
 * 
 * @author Gereon
 */
public class PhaseHandler extends BaseObserver
{
	private DependencyScheduler<BaseObserver> manager = null;
	
	private Iterator<Set<BaseObserver>> iterator = null;
	private ObserverHandler activeObservers = null;
	/**
	 * the constructor...
	 * @param parser the {@link TreeParser} that calls this {@link PhaseHandler}
	 */
	public PhaseHandler(TreeParser parser)
	{
		super(parser);
		this.manager = new DependencyScheduler<BaseObserver>();
	}

	/**
	 * get all observers
	 */
	public void prepare()
	{
		this.manager.addAll(ObserverRegister.getObservers());
		this.manager.reset();
		this.manager.dumpDependencyGraph(new File("dot/dependencySchedule.dot"));
		this.iterator = this.manager.iterator();
	}
	
	/**
	 * checks if there are child observers that have not been run
	 * @return true if all observers have been run
	 */
	public boolean isDone()
	{
		if (this.iterator == null) return false;
		return !this.iterator.hasNext();
	}
	
	/**
	 * select all child observers that can be run and call {@link #init()} on them
	 */
	public boolean init() 
	{
		this.activeObservers = new ObserverHandler(this.parser);
		this.activeObservers.addAll(this.iterator.next());
		
		return this.activeObservers.init();
	}
	
	/**
	 * give it to the child observers
	 */
	protected boolean ruleStart(Token nonterminal, List<Definition> nodes)
	{
		return this.activeObservers.startRule(nonterminal, nodes);
	}

	/**
	 * give it to the child observers
	 */
	protected boolean ruleEnd(Token nonterminal)
	{
		return this.activeObservers.endRule(nonterminal);
	}

	/**
	 * give it to the child observers
	 */
	public boolean edge(Token edgename, List<Definition> nodes)
	{
		return this.activeObservers.edge(edgename, nodes);
	}
	
	/**
	 * validate child observers
	 * 
	 * remove dependencies
	 */
	public boolean validate()
	{
		return this.activeObservers.validate();
	}
	
	/**
	 * get all observers that are currently active
	 * @return String of active observers
	 */
	public String getActiveObservers()
	{
		return this.activeObservers.getObservers().toString();
	}
}
